'use client';

import React, { useState, useEffect, useRef } from 'react';
import { attachmentAPI } from '@/lib/api';
import toast from 'react-hot-toast';
import { PaperClipIcon, TrashIcon, ArrowDownTrayIcon, DocumentIcon } from '@heroicons/react/24/outline';

interface FileAttachmentsProps {
  entityType: string;
  entityId: string;
}

const mimeIcons: Record<string, string> = {
  'application/pdf': '📄',
  'image/jpeg': '🖼️',
  'image/png': '🖼️',
  'image/gif': '🖼️',
  'image/webp': '🖼️',
  'application/msword': '📝',
  'application/vnd.openxmlformats-officedocument.wordprocessingml.document': '📝',
  'application/vnd.ms-excel': '📊',
  'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': '📊',
  'text/csv': '📊',
  'text/plain': '📃',
};

function formatFileSize(bytes: number): string {
  if (bytes < 1024) return bytes + ' B';
  if (bytes < 1048576) return (bytes / 1024).toFixed(1) + ' KB';
  return (bytes / 1048576).toFixed(1) + ' MB';
}

export default function FileAttachments({ entityType, entityId }: FileAttachmentsProps) {
  const [files, setFiles] = useState<any[]>([]);
  const [uploading, setUploading] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    if (entityId) loadFiles();
  }, [entityType, entityId]);

  const loadFiles = async () => {
    try {
      const res = await attachmentAPI.list({ entity_type: entityType, entity_id: entityId });
      setFiles(res.data?.data || []);
    } catch { /* ignore */ }
  };

  const handleUpload = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    const formData = new FormData();
    formData.append('file', file);
    formData.append('entity_type', entityType);
    formData.append('entity_id', entityId);

    setUploading(true);
    try {
      await attachmentAPI.upload(formData);
      toast.success('File uploaded');
      loadFiles();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Upload failed');
    }
    setUploading(false);
    if (inputRef.current) inputRef.current.value = '';
  };

  const handleDownload = async (attachment: any) => {
    try {
      const res = await attachmentAPI.download(attachment.id);
      const url = URL.createObjectURL(new Blob([res.data]));
      const a = document.createElement('a');
      a.href = url;
      a.download = attachment.original_name;
      a.click();
      URL.revokeObjectURL(url);
    } catch {
      toast.error('Download failed');
    }
  };

  const handleDelete = async (id: string) => {
    if (!confirm('Delete this file?')) return;
    try {
      await attachmentAPI.delete(id);
      toast.success('File deleted');
      loadFiles();
    } catch {
      toast.error('Delete failed');
    }
  };

  return (
    <div className="space-y-3">
      <div className="flex items-center justify-between">
        <h4 className="text-sm font-medium text-neutral-700 flex items-center gap-1.5">
          <PaperClipIcon className="h-4 w-4" /> Attachments ({files.length})
        </h4>
        <label className={`btn-secondary btn-sm cursor-pointer ${uploading ? 'opacity-50 pointer-events-none' : ''}`}>
          {uploading ? 'Uploading...' : 'Attach File'}
          <input
            ref={inputRef}
            type="file"
            className="hidden"
            onChange={handleUpload}
            accept=".pdf,.jpg,.jpeg,.png,.gif,.doc,.docx,.xls,.xlsx,.csv,.txt"
          />
        </label>
      </div>

      {files.length > 0 && (
        <div className="divide-y divide-neutral-100 border border-neutral-200 rounded-lg overflow-hidden">
          {files.map((f: any) => (
            <div key={f.id} className="flex items-center gap-3 px-3 py-2 hover:bg-neutral-50 text-sm">
              <span className="text-lg">{mimeIcons[f.mime_type] || '📎'}</span>
              <div className="flex-1 min-w-0">
                <p className="font-medium text-neutral-800 truncate">{f.original_name}</p>
                <p className="text-xs text-neutral-400">{formatFileSize(f.file_size)} · {new Date(f.created_at).toLocaleDateString()}</p>
              </div>
              <button onClick={() => handleDownload(f)} className="p-1 text-neutral-400 hover:text-neutral-700" title="Download">
                <ArrowDownTrayIcon className="h-4 w-4" />
              </button>
              <button onClick={() => handleDelete(f.id)} className="p-1 text-neutral-400 hover:text-red-600" title="Delete">
                <TrashIcon className="h-4 w-4" />
              </button>
            </div>
          ))}
        </div>
      )}

      {files.length === 0 && (
        <p className="text-xs text-neutral-400 text-center py-3 border border-dashed border-neutral-200 rounded-lg">
          No files attached. Click &quot;Attach File&quot; to upload.
        </p>
      )}
    </div>
  );
}
