'use client';

import React, { useState, useEffect, useRef } from 'react';
import Link from 'next/link';
import { usePathname } from 'next/navigation';
import { useAuth } from '@/context/AuthContext';
import { adminAPI, businessAPI } from '@/lib/api';
import clsx from 'clsx';
import toast from 'react-hot-toast';
import {
  HomeIcon,
  BookOpenIcon,
  DocumentTextIcon,
  ShoppingCartIcon,
  CubeIcon,
  UsersIcon,
  ChartBarIcon,
  CalculatorIcon,
  BuildingOfficeIcon,
  Cog6ToothIcon,
  ArrowRightOnRectangleIcon,
  Bars3Icon,
  XMarkIcon,
  BanknotesIcon,
  BellIcon,
  ArrowsRightLeftIcon,
  ChevronDownIcon,
  ClockIcon,
  BuildingStorefrontIcon,
  ChevronUpDownIcon,
} from '@heroicons/react/24/outline';

type NavItem = {
  name: string;
  href: string;
  icon: React.ForwardRefExoticComponent<any>;
  children?: { name: string; href: string; icon: React.ForwardRefExoticComponent<any> }[];
};

const navigation: NavItem[] = [
  { name: 'Dashboard', href: '/dashboard', icon: HomeIcon },
  { name: 'Transactions', href: '/dashboard/transactions', icon: ArrowsRightLeftIcon },
  { name: 'Accounting', href: '/dashboard/accounting', icon: BookOpenIcon },
  { name: 'Invoicing', href: '/dashboard/invoices', icon: DocumentTextIcon },
  {
    name: 'POS',
    href: '/dashboard/pos',
    icon: ShoppingCartIcon,
    children: [
      { name: 'POS Terminal', href: '/dashboard/pos', icon: ShoppingCartIcon },
      { name: 'Sales History', href: '/dashboard/pos-history', icon: ClockIcon },
    ],
  },
  { name: 'Payroll', href: '/dashboard/payroll', icon: BanknotesIcon },
  { name: 'Inventory', href: '/dashboard/inventory', icon: CubeIcon },
  { name: 'Contacts', href: '/dashboard/contacts', icon: UsersIcon },
  { name: 'Assets', href: '/dashboard/assets', icon: BuildingOfficeIcon },
  { name: 'Tax', href: '/dashboard/tax', icon: CalculatorIcon },
  { name: 'Reports', href: '/dashboard/reports', icon: ChartBarIcon },
  { name: 'Businesses', href: '/dashboard/businesses', icon: BuildingStorefrontIcon },
  { name: 'Settings', href: '/dashboard/settings', icon: Cog6ToothIcon },
];

export default function DashboardLayout({ children }: { children: React.ReactNode }) {
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [expandedMenus, setExpandedMenus] = useState<Record<string, boolean>>({});
  const [notifOpen, setNotifOpen] = useState(false);
  const [notifications, setNotifications] = useState<any[]>([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [businesses, setBusinesses] = useState<any[]>([]);
  const [bizSwitcherOpen, setBizSwitcherOpen] = useState(false);
  const [switchingBiz, setSwitchingBiz] = useState(false);
  const notifRef = useRef<HTMLDivElement>(null);
  const bizRef = useRef<HTMLDivElement>(null);
  const pathname = usePathname();
  const { user, logout, switchBusiness, isAdmin } = useAuth();

  // Auto-expand menus that contain the active route
  useEffect(() => {
    const expanded: Record<string, boolean> = {};
    navigation.forEach((item) => {
      if (item.children) {
        const isChildActive = item.children.some((child) => pathname?.startsWith(child.href));
        if (isChildActive) expanded[item.name] = true;
      }
    });
    setExpandedMenus((prev) => ({ ...prev, ...expanded }));
  }, [pathname]);

  const toggleMenu = (name: string) => {
    setExpandedMenus((prev) => ({ ...prev, [name]: !prev[name] }));
  };

  // Load notifications
  useEffect(() => {
    const loadNotifs = async () => {
      try {
        const res = await adminAPI.getNotifications();
        const notifs = res.data?.data || [];
        setNotifications(notifs);
        setUnreadCount(notifs.filter((n: any) => !n.is_read).length);
      } catch {
        // silently fail - notifications are not critical
      }
    };
    loadNotifs();
    const interval = setInterval(loadNotifs, 60000); // refresh every 60s
    return () => clearInterval(interval);
  }, []);

  // Close notification panel on outside click
  useEffect(() => {
    const handleClick = (e: MouseEvent) => {
      if (notifRef.current && !notifRef.current.contains(e.target as Node)) {
        setNotifOpen(false);
      }
      if (bizRef.current && !bizRef.current.contains(e.target as Node)) {
        setBizSwitcherOpen(false);
      }
    };
    if (notifOpen || bizSwitcherOpen) document.addEventListener('mousedown', handleClick);
    return () => document.removeEventListener('mousedown', handleClick);
  }, [notifOpen, bizSwitcherOpen]);

  // Load businesses for business switcher
  useEffect(() => {
    const loadBusinesses = async () => {
      try {
        const res = await businessAPI.list();
        setBusinesses(res.data?.data || []);
      } catch {
        // silently fail
      }
    };
    loadBusinesses();
  }, []);

  const handleSwitchBusiness = async (companyId: string) => {
    if (companyId === user?.company?.id) {
      setBizSwitcherOpen(false);
      return;
    }
    setSwitchingBiz(true);
    try {
      await switchBusiness(companyId);
    } catch {
      toast.error('Failed to switch business');
      setSwitchingBiz(false);
    }
  };

  const markRead = async (id: string) => {
    try {
      await adminAPI.markNotificationRead(id);
      setNotifications((prev) =>
        prev.map((n) => (n.id === id ? { ...n, is_read: true } : n))
      );
      setUnreadCount((prev) => Math.max(0, prev - 1));
    } catch {
      toast.error('Failed to mark notification as read');
    }
  };

  return (
    <div className="flex h-screen overflow-hidden bg-neutral-50">
      {/* Mobile sidebar overlay */}
      {sidebarOpen && (
        <div
          className="fixed inset-0 z-40 bg-black/50 lg:hidden"
          onClick={() => setSidebarOpen(false)}
        />
      )}

      {/* Sidebar */}
      <aside
        className={clsx(
          'fixed inset-y-0 left-0 z-50 w-64 bg-neutral-900 transform transition-transform duration-200 ease-in-out lg:translate-x-0 lg:static lg:inset-0 flex flex-col',
          sidebarOpen ? 'translate-x-0' : '-translate-x-full'
        )}
      >
        {/* Logo */}
        <div className="flex items-center justify-between h-16 px-6 border-b border-neutral-800">
          <Link href="/dashboard" className="flex items-center gap-2">
            <div className="h-7 w-7 bg-white rounded-sm flex items-center justify-center">
              <BookOpenIcon className="h-4 w-4 text-neutral-900" />
            </div>
            <span className="text-lg font-semibold text-white">CashBook</span>
          </Link>
          <button
            className="lg:hidden text-neutral-400 hover:text-white"
            onClick={() => setSidebarOpen(false)}
          >
            <XMarkIcon className="h-6 w-6" />
          </button>
        </div>

        {/* Business Switcher */}
        <div className="px-3 py-3 border-b border-neutral-800" ref={bizRef}>
          <p className="text-[10px] text-neutral-500 uppercase tracking-wider mb-1 px-3">Business</p>
          <div className="relative">
            <button
              onClick={() => setBizSwitcherOpen(!bizSwitcherOpen)}
              disabled={switchingBiz}
              className="w-full flex items-center gap-2 px-3 py-2 rounded-md hover:bg-neutral-800 transition-colors text-left"
            >
              <BuildingStorefrontIcon className="h-4 w-4 text-neutral-400 flex-shrink-0" />
              <span className="text-sm font-medium text-neutral-200 truncate flex-1">
                {switchingBiz ? 'Switching...' : (user?.company?.name || 'My Business')}
              </span>
              {businesses.length > 1 && (
                <ChevronUpDownIcon className="h-4 w-4 text-neutral-500 flex-shrink-0" />
              )}
            </button>

            {/* Business Dropdown */}
            {bizSwitcherOpen && businesses.length > 1 && (
              <div className="absolute left-0 right-0 top-full mt-1 bg-neutral-800 rounded-md border border-neutral-700 shadow-lg z-50 overflow-hidden max-h-60 overflow-y-auto">
                {businesses.map((biz: any) => (
                  <button
                    key={biz.id}
                    onClick={() => handleSwitchBusiness(biz.id)}
                    className={clsx(
                      'w-full text-left px-3 py-2 text-sm hover:bg-neutral-700 transition-colors flex items-center gap-2',
                      biz.id === user?.company?.id ? 'text-white bg-neutral-700/50' : 'text-neutral-300'
                    )}
                  >
                    <BuildingStorefrontIcon className="h-4 w-4 flex-shrink-0" />
                    <span className="truncate flex-1">{biz.name}</span>
                    {biz.id === user?.company?.id && (
                      <span className="text-[10px] bg-green-600 text-white px-1.5 py-0.5 rounded">Active</span>
                    )}
                  </button>
                ))}
                {isAdmin && (
                  <Link
                    href="/dashboard/businesses"
                    onClick={() => setBizSwitcherOpen(false)}
                    className="block px-3 py-2 text-xs text-center text-neutral-400 hover:text-white border-t border-neutral-700 hover:bg-neutral-700 transition-colors"
                  >
                    + Manage Businesses
                  </Link>
                )}
              </div>
            )}
          </div>
        </div>

        {/* Navigation */}
        <nav className="flex-1 overflow-y-auto px-3 py-4 space-y-0.5">
          {navigation.map((item) => {
            const isActive =
              item.href === '/dashboard'
                ? pathname === '/dashboard'
                : item.children
                  ? item.children.some((child) => pathname?.startsWith(child.href))
                  : pathname?.startsWith(item.href);

            // Item with children - render collapsible submenu
            if (item.children) {
              const isExpanded = expandedMenus[item.name] || false;
              return (
                <div key={item.name}>
                  <button
                    onClick={() => toggleMenu(item.name)}
                    className={clsx(
                      'sidebar-link w-full justify-between',
                      isActive && 'active'
                    )}
                  >
                    <span className="flex items-center gap-3">
                      <item.icon className="h-5 w-5 flex-shrink-0" />
                      {item.name}
                    </span>
                    <ChevronDownIcon
                      className={clsx(
                        'h-4 w-4 flex-shrink-0 transition-transform duration-200',
                        isExpanded && 'rotate-180'
                      )}
                    />
                  </button>
                  <div
                    className={clsx(
                      'overflow-hidden transition-all duration-200 ease-in-out',
                      isExpanded ? 'max-h-40 opacity-100' : 'max-h-0 opacity-0'
                    )}
                  >
                    <div className="ml-4 mt-0.5 space-y-0.5 border-l border-neutral-700 pl-3">
                      {item.children.map((child) => {
                        const isChildActive = pathname === child.href || pathname?.startsWith(child.href + '/');
                        return (
                          <Link
                            key={child.href}
                            href={child.href}
                            className={clsx(
                              'sidebar-link text-sm py-1.5',
                              isChildActive && 'active'
                            )}
                            onClick={() => setSidebarOpen(false)}
                          >
                            <child.icon className="h-4 w-4 flex-shrink-0" />
                            {child.name}
                          </Link>
                        );
                      })}
                    </div>
                  </div>
                </div>
              );
            }

            // Regular item
            return (
              <Link
                key={item.name}
                href={item.href}
                className={clsx('sidebar-link', isActive && 'active')}
                onClick={() => setSidebarOpen(false)}
              >
                <item.icon className="h-5 w-5 flex-shrink-0" />
                {item.name}
              </Link>
            );
          })}
        </nav>

        {/* User section */}
        <div className="p-4 border-t border-neutral-800">
          <div className="flex items-center gap-3">
            <div className="w-8 h-8 rounded-md bg-neutral-700 flex items-center justify-center text-xs font-semibold text-neutral-200">
              {user?.first_name?.[0]}{user?.last_name?.[0]}
            </div>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-medium text-neutral-200 truncate">
                {user?.first_name} {user?.last_name}
              </p>
              <p className="text-xs text-neutral-500 capitalize">{user?.role}</p>
            </div>
            <button
              onClick={logout}
              className="text-neutral-500 hover:text-white transition-colors"
              title="Logout"
            >
              <ArrowRightOnRectangleIcon className="h-5 w-5" />
            </button>
          </div>
        </div>
      </aside>

      {/* Main content */}
      <div className="flex-1 flex flex-col overflow-hidden">
        {/* Top bar */}
        <header className="h-14 bg-white border-b border-neutral-200 flex items-center px-4 lg:px-8 gap-4 no-print">
          <button
            className="lg:hidden text-neutral-500 hover:text-neutral-700"
            onClick={() => setSidebarOpen(true)}
          >
            <Bars3Icon className="h-6 w-6" />
          </button>
          <div className="flex-1" />
          <div className="flex items-center gap-4">
            {/* Notification Bell */}
            <div className="relative" ref={notifRef}>
              <button
                onClick={() => setNotifOpen(!notifOpen)}
                className="relative p-2 rounded-md hover:bg-neutral-100 text-neutral-500 hover:text-neutral-700 transition-colors"
              >
                <BellIcon className="h-5 w-5" />
                {unreadCount > 0 && (
                  <span className="absolute top-1 right-1 h-4 w-4 bg-red-500 rounded-full text-[10px] font-bold text-white flex items-center justify-center">
                    {unreadCount > 9 ? '9+' : unreadCount}
                  </span>
                )}
              </button>

              {/* Notification Dropdown */}
              {notifOpen && (
                <div className="absolute right-0 top-full mt-2 w-80 bg-white rounded-md border border-neutral-200 shadow-lg z-50 overflow-hidden">
                  <div className="px-4 py-3 border-b border-neutral-200 flex items-center justify-between">
                    <h4 className="text-sm font-semibold text-neutral-900">Notifications</h4>
                    {unreadCount > 0 && (
                      <span className="text-xs text-neutral-500">{unreadCount} unread</span>
                    )}
                  </div>
                  <div className="max-h-80 overflow-y-auto">
                    {notifications.length === 0 ? (
                      <p className="text-sm text-neutral-400 text-center py-8">No notifications</p>
                    ) : (
                      notifications.slice(0, 10).map((n: any) => (
                        <div
                          key={n.id}
                          className={clsx(
                            'px-4 py-3 border-b border-neutral-100 last:border-0 hover:bg-neutral-50 transition-colors cursor-pointer',
                            !n.is_read && 'bg-blue-50/50'
                          )}
                          onClick={() => !n.is_read && markRead(n.id)}
                        >
                          <div className="flex items-start gap-2">
                            {!n.is_read && (
                              <span className="mt-1.5 h-2 w-2 rounded-full bg-blue-500 flex-shrink-0" />
                            )}
                            <div className="flex-1 min-w-0">
                              <p className={clsx('text-xs', !n.is_read ? 'font-semibold text-neutral-900' : 'text-neutral-700')}>
                                {n.title}
                              </p>
                              <p className="text-xs text-neutral-500 mt-0.5 line-clamp-2">{n.message}</p>
                              <p className="text-[10px] text-neutral-400 mt-1">
                                {new Date(n.created_at).toLocaleDateString('en-GH', { month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' })}
                              </p>
                            </div>
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                  <Link
                    href="/dashboard/settings"
                    onClick={() => setNotifOpen(false)}
                    className="block px-4 py-2.5 text-xs text-center font-medium text-neutral-600 hover:text-neutral-900 border-t border-neutral-200 hover:bg-neutral-50 transition-colors"
                  >
                    View all notifications
                  </Link>
                </div>
              )}
            </div>
            <span className="text-sm text-neutral-500">
              {new Date().toLocaleDateString('en-GH', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric',
              })}
            </span>
          </div>
        </header>

        {/* Page content */}
        <main className="flex-1 overflow-y-auto p-4 lg:p-8">
          {children}
        </main>
      </div>
    </div>
  );
}
