'use client';

import React, { useState, useEffect, useCallback } from 'react';
import { transactionAPI, accountingAPI, contactAPI } from '@/lib/api';
import { formatCurrency } from '@/lib/utils';
import { PageLoading, Modal, StatusBadge, StatCard, EmptyState } from '@/components/ui';
import FileAttachments from '@/components/ui/FileAttachments';
import toast from 'react-hot-toast';
import {
  ArrowsRightLeftIcon,
  ArrowUpCircleIcon,
  ArrowDownCircleIcon,
  PlusIcon,
  FunnelIcon,
  MagnifyingGlassIcon,
  EyeIcon,
  TrashIcon,
  BanknotesIcon,
  DevicePhoneMobileIcon,
  CreditCardIcon,
  BuildingLibraryIcon,
} from '@heroicons/react/24/outline';

const CATEGORIES_INCOME = [
  'Sales Revenue', 'Service Revenue', 'Interest Income', 'Rental Income',
  'Commission', 'Consulting Fees', 'Investment Returns', 'Other Income'
];

const CATEGORIES_EXPENSE = [
  'Office Supplies', 'Rent', 'Utilities', 'Internet & Communication',
  'Transportation', 'Marketing & Advertising', 'Salaries & Wages',
  'Insurance', 'Repairs & Maintenance', 'Bank Charges',
  'Mobile Money Charges', 'Fuel & Energy', 'Food & Meals',
  'Subscriptions', 'Professional Fees', 'Equipment', 'Miscellaneous'
];

const PAYMENT_METHODS = [
  { value: 'cash', label: 'Cash', icon: BanknotesIcon },
  { value: 'mobile_money', label: 'Mobile Money', icon: DevicePhoneMobileIcon },
  { value: 'bank_transfer', label: 'Bank Transfer', icon: BuildingLibraryIcon },
  { value: 'card', label: 'Card', icon: CreditCardIcon },
  { value: 'cheque', label: 'Cheque', icon: BanknotesIcon },
];

export default function TransactionsPage() {
  const [loading, setLoading] = useState(true);
  const [transactions, setTransactions] = useState<any[]>([]);
  const [total, setTotal] = useState(0);
  const [page, setPage] = useState(1);
  const [summary, setSummary] = useState<any>(null);

  // Filters
  const [filterType, setFilterType] = useState('');
  const [filterMethod, setFilterMethod] = useState('');
  const [dateFrom, setDateFrom] = useState('');
  const [dateTo, setDateTo] = useState('');
  const [search, setSearch] = useState('');
  const [showFilters, setShowFilters] = useState(false);

  // Create modal
  const [showCreate, setShowCreate] = useState(false);
  const [createType, setCreateType] = useState<'income' | 'expense'>('income');
  const [saving, setSaving] = useState(false);

  // View modal
  const [showView, setShowView] = useState(false);
  const [viewTxn, setViewTxn] = useState<any>(null);

  // Form state
  const [form, setForm] = useState({
    transaction_date: new Date().toISOString().split('T')[0],
    amount: '',
    description: '',
    category: '',
    payment_method: 'cash',
    mobile_money_provider: '',
    mobile_money_number: '',
    reference: '',
    contact_id: '',
    account_id: '',
    notes: '',
  });

  // Reference data
  const [accounts, setAccounts] = useState<any[]>([]);
  const [contacts, setContacts] = useState<any[]>([]);

  const loadTransactions = useCallback(async () => {
    try {
      const params: any = { page, per_page: 20 };
      if (filterType) params.type = filterType;
      if (filterMethod) params.payment_method = filterMethod;
      if (dateFrom) params.date_from = dateFrom;
      if (dateTo) params.date_to = dateTo;
      if (search) params.search = search;

      const [txnRes, sumRes] = await Promise.all([
        transactionAPI.list(params),
        transactionAPI.summary({
          date_from: dateFrom || new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0],
          date_to: dateTo || new Date().toISOString().split('T')[0],
        }),
      ]);

      setTransactions(txnRes.data?.data || []);
      setTotal(txnRes.data?.pagination?.total || 0);
      setSummary(sumRes.data?.data || null);
    } catch (err) {
      console.error(err);
    }
    setLoading(false);
  }, [page, filterType, filterMethod, dateFrom, dateTo, search]);

  useEffect(() => {
    loadTransactions();
  }, [loadTransactions]);

  useEffect(() => {
    const loadRefs = async () => {
      try {
        const [acctRes, contRes] = await Promise.all([
          accountingAPI.getAccounts(),
          contactAPI.list({ limit: 200 }),
        ]);
        setAccounts(acctRes.data?.data || []);
        setContacts(contRes.data?.data || []);
      } catch {}
    };
    loadRefs();
  }, []);

  const openCreate = (type: 'income' | 'expense') => {
    setCreateType(type);
    setForm({
      transaction_date: new Date().toISOString().split('T')[0],
      amount: '',
      description: '',
      category: '',
      payment_method: 'cash',
      mobile_money_provider: '',
      mobile_money_number: '',
      reference: '',
      contact_id: '',
      account_id: '',
      notes: '',
    });
    setShowCreate(true);
  };

  const handleCreate = async () => {
    if (!form.amount || !form.description) {
      toast.error('Amount and description are required');
      return;
    }
    setSaving(true);
    try {
      await transactionAPI.create({
        transaction_type: createType,
        ...form,
        amount: parseFloat(form.amount),
      });
      toast.success(`${createType === 'income' ? 'Income' : 'Expense'} recorded`);
      setShowCreate(false);
      loadTransactions();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Failed to record transaction');
    }
    setSaving(false);
  };

  const viewTransaction = async (id: string) => {
    try {
      const { data } = await transactionAPI.get(id);
      setViewTxn(data.data);
      setShowView(true);
    } catch {
      toast.error('Failed to load transaction');
    }
  };

  const voidTransaction = async (id: string) => {
    if (!confirm('Are you sure you want to void this transaction?')) return;
    try {
      await transactionAPI.void(id);
      toast.success('Transaction voided');
      loadTransactions();
      setShowView(false);
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Failed to void');
    }
  };

  const totalPages = Math.ceil(total / 20);
  const netIncome = (summary?.total_income || 0) - (summary?.total_expenses || 0);
  const categories = createType === 'income' ? CATEGORIES_INCOME : CATEGORIES_EXPENSE;

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-neutral-900">Transactions</h1>
          <p className="text-sm text-neutral-500 mt-1">Record and track income & expenses</p>
        </div>
        <div className="flex gap-3">
          <button className="btn-primary flex items-center gap-2" onClick={() => openCreate('income')}>
            <ArrowDownCircleIcon className="h-4 w-4" />
            Record Income
          </button>
          <button className="btn-danger flex items-center gap-2" onClick={() => openCreate('expense')}>
            <ArrowUpCircleIcon className="h-4 w-4" />
            Record Expense
          </button>
        </div>
      </div>

      {/* Summary Cards */}
      {summary && (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
          <StatCard
            title="Total Income"
            value={formatCurrency(summary.total_income)}
            subtitle={`${summary.income_count} transactions`}
            icon={<ArrowDownCircleIcon className="h-5 w-5 text-emerald-600" />}
          />
          <StatCard
            title="Total Expenses"
            value={formatCurrency(summary.total_expenses)}
            subtitle={`${summary.expense_count} transactions`}
            icon={<ArrowUpCircleIcon className="h-5 w-5 text-red-600" />}
          />
          <StatCard
            title="Net Income"
            value={formatCurrency(netIncome)}
            subtitle={netIncome >= 0 ? 'Profit' : 'Loss'}
            icon={<ArrowsRightLeftIcon className="h-5 w-5" />}
          />
          <StatCard
            title="Cash Balance"
            value={formatCurrency(summary.cash_balance)}
            subtitle={`MoMo: ${formatCurrency(summary.momo_balance)}`}
            icon={<BanknotesIcon className="h-5 w-5" />}
          />
        </div>
      )}

      {/* Search & Filters */}
      <div className="card p-4">
        <div className="flex flex-wrap items-center gap-3">
          <div className="relative flex-1 min-w-[200px]">
            <MagnifyingGlassIcon className="h-5 w-5 absolute left-3 top-2.5 text-neutral-400" />
            <input
              className="input pl-10"
              placeholder="Search transactions..."
              value={search}
              onChange={(e) => { setSearch(e.target.value); setPage(1); }}
            />
          </div>
          <select
            className="select w-auto"
            value={filterType}
            onChange={(e) => { setFilterType(e.target.value); setPage(1); }}
          >
            <option value="">All Types</option>
            <option value="income">Income</option>
            <option value="expense">Expense</option>
          </select>
          <select
            className="select w-auto"
            value={filterMethod}
            onChange={(e) => { setFilterMethod(e.target.value); setPage(1); }}
          >
            <option value="">All Methods</option>
            <option value="cash">Cash</option>
            <option value="mobile_money">Mobile Money</option>
            <option value="bank_transfer">Bank Transfer</option>
            <option value="card">Card</option>
            <option value="cheque">Cheque</option>
          </select>
          <button className="btn-secondary flex items-center gap-1" onClick={() => setShowFilters(!showFilters)}>
            <FunnelIcon className="h-4 w-4" /> Date Range
          </button>
        </div>
        {showFilters && (
          <div className="flex items-center gap-3 mt-3 pt-3 border-t border-neutral-200">
            <div>
              <label className="text-xs text-neutral-500">From</label>
              <input type="date" className="input" value={dateFrom} onChange={(e) => { setDateFrom(e.target.value); setPage(1); }} />
            </div>
            <div>
              <label className="text-xs text-neutral-500">To</label>
              <input type="date" className="input" value={dateTo} onChange={(e) => { setDateTo(e.target.value); setPage(1); }} />
            </div>
            <button className="btn-secondary btn-sm mt-4" onClick={() => { setDateFrom(''); setDateTo(''); setPage(1); }}>Clear</button>
          </div>
        )}
      </div>

      {/* Transactions Table */}
      <div className="card">
        <div className="table-container">
          <table className="table">
            <thead>
              <tr>
                <th>Txn #</th>
                <th>Date</th>
                <th>Type</th>
                <th>Description</th>
                <th>Category</th>
                <th>Method</th>
                <th className="text-right">Amount</th>
                <th>Status</th>
                <th></th>
              </tr>
            </thead>
            <tbody>
              {transactions.length === 0 ? (
                <tr><td colSpan={9} className="text-center py-8">
                  <EmptyState
                    icon={<ArrowsRightLeftIcon className="h-12 w-12" />}
                    title="No transactions found"
                    description="Record your first income or expense to get started"
                  />
                </td></tr>
              ) : (
                transactions.map((txn: any) => (
                  <tr key={txn.id} className="hover:bg-neutral-50">
                    <td className="font-mono text-xs font-medium">{txn.transaction_number}</td>
                    <td className="text-sm">{new Date(txn.transaction_date).toLocaleDateString('en-GH', { day: 'numeric', month: 'short', year: 'numeric' })}</td>
                    <td>
                      <span className={`inline-flex items-center gap-1 text-xs font-medium ${txn.transaction_type === 'income' ? 'text-emerald-700' : 'text-red-700'}`}>
                        {txn.transaction_type === 'income' ? <ArrowDownCircleIcon className="h-4 w-4" /> : <ArrowUpCircleIcon className="h-4 w-4" />}
                        {txn.transaction_type}
                      </span>
                    </td>
                    <td className="text-sm max-w-[200px] truncate">{txn.description}</td>
                    <td className="text-xs text-neutral-500">{txn.category || '-'}</td>
                    <td className="text-xs capitalize">{txn.payment_method?.replace(/_/g, ' ')}</td>
                    <td className={`text-right font-semibold ${txn.transaction_type === 'income' ? 'text-emerald-700' : 'text-red-700'}`}>
                      {txn.transaction_type === 'income' ? '+' : '-'}{formatCurrency(txn.amount)}
                    </td>
                    <td><StatusBadge status={txn.status} /></td>
                    <td>
                      <button onClick={() => viewTransaction(txn.id)} className="text-neutral-400 hover:text-neutral-700">
                        <EyeIcon className="h-4 w-4" />
                      </button>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        {totalPages > 1 && (
          <div className="flex items-center justify-between px-4 py-3 border-t border-neutral-200">
            <p className="text-sm text-neutral-500">
              Showing {(page - 1) * 20 + 1}-{Math.min(page * 20, total)} of {total}
            </p>
            <div className="flex gap-2">
              <button className="btn-secondary btn-sm" disabled={page <= 1} onClick={() => setPage(page - 1)}>Previous</button>
              <button className="btn-secondary btn-sm" disabled={page >= totalPages} onClick={() => setPage(page + 1)}>Next</button>
            </div>
          </div>
        )}
      </div>

      {/* Create Transaction Modal */}
      <Modal open={showCreate} onClose={() => setShowCreate(false)} title={`Record ${createType === 'income' ? 'Income' : 'Expense'}`} size="lg">
        <div className="space-y-4">
          {/* Type Toggle */}
          <div className="flex gap-2 p-1 bg-neutral-100 rounded-lg">
            <button
              className={`flex-1 py-2 px-4 text-sm font-medium rounded-md transition-all ${createType === 'income' ? 'bg-emerald-600 text-white shadow-sm' : 'text-neutral-600 hover:text-neutral-900'}`}
              onClick={() => setCreateType('income')}
            >
              <ArrowDownCircleIcon className="h-4 w-4 inline mr-1" /> Income
            </button>
            <button
              className={`flex-1 py-2 px-4 text-sm font-medium rounded-md transition-all ${createType === 'expense' ? 'bg-red-600 text-white shadow-sm' : 'text-neutral-600 hover:text-neutral-900'}`}
              onClick={() => setCreateType('expense')}
            >
              <ArrowUpCircleIcon className="h-4 w-4 inline mr-1" /> Expense
            </button>
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Date *</label>
              <input type="date" className="input" value={form.transaction_date}
                onChange={(e) => setForm({ ...form, transaction_date: e.target.value })} />
            </div>
            <div>
              <label className="label">Amount (GHS) *</label>
              <input type="number" step="0.01" min="0.01" className="input text-lg font-bold" placeholder="0.00"
                value={form.amount} onChange={(e) => setForm({ ...form, amount: e.target.value })} autoFocus />
            </div>
          </div>

          <div>
            <label className="label">Description *</label>
            <input className="input" placeholder={createType === 'income' ? 'e.g., Payment from client' : 'e.g., Office rent for January'}
              value={form.description} onChange={(e) => setForm({ ...form, description: e.target.value })} />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Category</label>
              <select className="select" value={form.category} onChange={(e) => setForm({ ...form, category: e.target.value })}>
                <option value="">Select category</option>
                {categories.map((c) => <option key={c} value={c}>{c}</option>)}
              </select>
            </div>
            <div>
              <label className="label">Payment Method *</label>
              <div className="grid grid-cols-3 gap-2">
                {PAYMENT_METHODS.slice(0, 3).map((m) => (
                  <button key={m.value} type="button"
                    className={`p-2 rounded-lg border-2 text-center text-xs font-medium transition-all ${form.payment_method === m.value ? 'border-neutral-900 bg-neutral-50' : 'border-neutral-200 hover:border-neutral-300'}`}
                    onClick={() => setForm({ ...form, payment_method: m.value })}
                  >
                    <m.icon className="h-4 w-4 mx-auto mb-0.5" />
                    {m.label}
                  </button>
                ))}
              </div>
              <div className="grid grid-cols-2 gap-2 mt-2">
                {PAYMENT_METHODS.slice(3).map((m) => (
                  <button key={m.value} type="button"
                    className={`p-2 rounded-lg border-2 text-center text-xs font-medium transition-all ${form.payment_method === m.value ? 'border-neutral-900 bg-neutral-50' : 'border-neutral-200 hover:border-neutral-300'}`}
                    onClick={() => setForm({ ...form, payment_method: m.value })}
                  >
                    {m.label}
                  </button>
                ))}
              </div>
            </div>
          </div>

          {form.payment_method === 'mobile_money' && (
            <div className="grid grid-cols-2 gap-4">
              <div>
                <label className="label">MoMo Provider</label>
                <select className="select" value={form.mobile_money_provider}
                  onChange={(e) => setForm({ ...form, mobile_money_provider: e.target.value })}>
                  <option value="">Select provider</option>
                  <option value="mtn">MTN Mobile Money</option>
                  <option value="vodafone">Vodafone Cash</option>
                  <option value="airteltigo">AirtelTigo Money</option>
                </select>
              </div>
              <div>
                <label className="label">MoMo Number</label>
                <input className="input" placeholder="024XXXXXXX"
                  value={form.mobile_money_number} onChange={(e) => setForm({ ...form, mobile_money_number: e.target.value })} />
              </div>
            </div>
          )}

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Reference / Receipt #</label>
              <input className="input" placeholder="e.g., INV-001, Receipt #123"
                value={form.reference} onChange={(e) => setForm({ ...form, reference: e.target.value })} />
            </div>
            <div>
              <label className="label">Contact</label>
              <select className="select" value={form.contact_id}
                onChange={(e) => setForm({ ...form, contact_id: e.target.value })}>
                <option value="">No contact</option>
                {contacts.map((c: any) => <option key={c.id} value={c.id}>{c.name}</option>)}
              </select>
            </div>
          </div>

          <div>
            <label className="label">Account</label>
            <select className="select" value={form.account_id}
              onChange={(e) => setForm({ ...form, account_id: e.target.value })}>
              <option value="">Default ({createType === 'income' ? 'Other Income 4300' : 'Miscellaneous Expense 5900'})</option>
              {accounts.filter((a: any) => createType === 'income' ? a.account_type === 'revenue' : a.account_type === 'expense').map((a: any) => (
                <option key={a.id} value={a.id}>{a.account_code} - {a.account_name}</option>
              ))}
            </select>
          </div>

          <div>
            <label className="label">Notes</label>
            <textarea className="input" rows={2} placeholder="Additional notes..."
              value={form.notes} onChange={(e) => setForm({ ...form, notes: e.target.value })} />
          </div>

          <div className="flex justify-end gap-3 pt-4 border-t border-neutral-200">
            <button className="btn-secondary" onClick={() => setShowCreate(false)}>Cancel</button>
            <button
              className={createType === 'income' ? 'btn-success' : 'btn-danger'}
              onClick={handleCreate}
              disabled={saving}
            >
              {saving ? 'Saving...' : `Record ${createType === 'income' ? 'Income' : 'Expense'}`}
            </button>
          </div>
        </div>
      </Modal>

      {/* View Transaction Modal */}
      <Modal open={showView} onClose={() => setShowView(false)} title="Transaction Details" size="lg">
        {viewTxn && (
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-neutral-500 font-mono">{viewTxn.transaction_number}</p>
                <p className={`text-lg font-bold ${viewTxn.transaction_type === 'income' ? 'text-emerald-700' : 'text-red-700'}`}>
                  {viewTxn.transaction_type === 'income' ? '+' : '-'}{formatCurrency(viewTxn.amount)}
                </p>
              </div>
              <div className="flex items-center gap-3">
                <StatusBadge status={viewTxn.status} />
                <span className={`inline-flex items-center gap-1 px-3 py-1 rounded-full text-xs font-medium ${viewTxn.transaction_type === 'income' ? 'bg-emerald-50 text-emerald-700' : 'bg-red-50 text-red-700'}`}>
                  {viewTxn.transaction_type === 'income' ? <ArrowDownCircleIcon className="h-3 w-3" /> : <ArrowUpCircleIcon className="h-3 w-3" />}
                  {viewTxn.transaction_type}
                </span>
              </div>
            </div>

            <div className="grid grid-cols-2 gap-4 bg-neutral-50 rounded-lg p-4 text-sm">
              <div><span className="text-neutral-500">Date:</span> <span className="font-medium">{new Date(viewTxn.transaction_date).toLocaleDateString('en-GH', { weekday: 'short', day: 'numeric', month: 'long', year: 'numeric' })}</span></div>
              <div><span className="text-neutral-500">Method:</span> <span className="font-medium capitalize">{viewTxn.payment_method?.replace(/_/g, ' ')}</span></div>
              <div><span className="text-neutral-500">Category:</span> <span className="font-medium">{viewTxn.category || 'Uncategorized'}</span></div>
              <div><span className="text-neutral-500">Reference:</span> <span className="font-medium">{viewTxn.reference || '-'}</span></div>
              {viewTxn.contact_name && <div><span className="text-neutral-500">Contact:</span> <span className="font-medium">{viewTxn.contact_name}</span></div>}
              {viewTxn.account_name && <div><span className="text-neutral-500">Account:</span> <span className="font-medium">{viewTxn.account_code} - {viewTxn.account_name}</span></div>}
              <div className="col-span-2"><span className="text-neutral-500">Description:</span> <span className="font-medium">{viewTxn.description}</span></div>
              {viewTxn.notes && <div className="col-span-2"><span className="text-neutral-500">Notes:</span> <span className="font-medium">{viewTxn.notes}</span></div>}
              <div><span className="text-neutral-500">Created by:</span> <span className="font-medium">{viewTxn.created_by_name || '-'}</span></div>
              <div><span className="text-neutral-500">Created:</span> <span className="font-medium">{new Date(viewTxn.created_at).toLocaleString('en-GH')}</span></div>
            </div>

            {/* File Attachments */}
            <FileAttachments entityType="transaction" entityId={viewTxn.id} />

            <div className="flex justify-between pt-4 border-t border-neutral-200">
              {viewTxn.status !== 'voided' && (
                <button className="btn-danger btn-sm flex items-center gap-1" onClick={() => voidTransaction(viewTxn.id)}>
                  <TrashIcon className="h-4 w-4" /> Void Transaction
                </button>
              )}
              <button className="btn-secondary ml-auto" onClick={() => setShowView(false)}>Close</button>
            </div>
          </div>
        )}
      </Modal>
    </div>
  );
}
