'use client';

import React, { useState, useEffect } from 'react';
import { adminAPI } from '@/lib/api';
import { useAuth } from '@/context/AuthContext';
import { ghanaRegions } from '@/lib/utils';
import { Modal, ConfirmDialog, PageLoading, StatusBadge } from '@/components/ui';
import toast from 'react-hot-toast';

export default function SettingsPage() {
  const { user } = useAuth();
  const [tab, setTab] = useState('company');
  const [loading, setLoading] = useState(false);

  // Company
  const [company, setCompany] = useState<any>(null);
  const [companyForm, setCompanyForm] = useState<any>({});

  // Users
  const [users, setUsers] = useState<any[]>([]);
  const [showUserModal, setShowUserModal] = useState(false);
  const [userForm, setUserForm] = useState<any>({});
  const [editUserId, setEditUserId] = useState<string | null>(null);
  const [confirmDeactivate, setConfirmDeactivate] = useState<string | null>(null);

  // Bank Accounts
  const [bankAccounts, setBankAccounts] = useState<any[]>([]);
  const [showBankModal, setShowBankModal] = useState(false);
  const [bankForm, setBankForm] = useState<any>({});

  // Notifications
  const [notifications, setNotifications] = useState<any[]>([]);

  const loadData = async () => {
    setLoading(true);
    try {
      if (tab === 'company') {
        const res = await adminAPI.getCompany();
        setCompany(res.data.data);
        setCompanyForm(res.data.data || {});
      } else if (tab === 'users') {
        const res = await adminAPI.listUsers();
        setUsers(res.data.data || []);
      } else if (tab === 'bank') {
        const res = await adminAPI.listBankAccounts();
        setBankAccounts(res.data.data || []);
      } else if (tab === 'notifications') {
        const res = await adminAPI.getNotifications();
        setNotifications(res.data.data || []);
      }
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error loading data');
    }
    setLoading(false);
  };

  useEffect(() => { loadData(); }, [tab]);

  const saveCompany = async () => {
    try {
      await adminAPI.updateCompany(companyForm);
      toast.success('Company settings updated');
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error saving');
    }
  };

  const saveUser = async () => {
    try {
      if (editUserId) {
        await adminAPI.updateUser(editUserId, userForm);
        toast.success('User updated');
      } else {
        await adminAPI.createUser(userForm);
        toast.success('User created');
      }
      setShowUserModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error saving user');
    }
  };

  const deactivateUser = async () => {
    if (!confirmDeactivate) return;
    try {
      await adminAPI.deactivateUser(confirmDeactivate);
      toast.success('User deactivated');
      setConfirmDeactivate(null);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const saveBankAccount = async () => {
    try {
      await adminAPI.createBankAccount(bankForm);
      toast.success('Bank account added');
      setShowBankModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error saving');
    }
  };

  const markRead = async (id: string) => {
    try {
      await adminAPI.markNotificationRead(id);
      loadData();
    } catch (err: any) {
      toast.error('Error');
    }
  };

  const tabs = [
    { key: 'company', label: 'Company' },
    { key: 'users', label: 'Users' },
    { key: 'bank', label: 'Bank Accounts' },
    { key: 'notifications', label: 'Notifications' },
  ];

  return (
    <div className="space-y-6">
      <h1 className="text-2xl font-bold text-gray-900">Settings</h1>

      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {tabs.map((t) => (
            <button key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500'
              }`}
              onClick={() => setTab(t.key)}
            >{t.label}</button>
          ))}
        </nav>
      </div>

      {loading && <PageLoading />}

      {/* Company Settings */}
      {!loading && tab === 'company' && (
        <div className="card p-6 max-w-2xl">
          <h3 className="font-semibold text-lg mb-4">Company Information</h3>
          <div className="grid grid-cols-2 gap-4">
            <div className="col-span-2">
              <label className="label">Company Name</label>
              <input className="input" value={companyForm.name || ''} onChange={(e) => setCompanyForm({ ...companyForm, name: e.target.value })} />
            </div>
            <div>
              <label className="label">TIN Number</label>
              <input className="input" value={companyForm.tin_number || ''} onChange={(e) => setCompanyForm({ ...companyForm, tin_number: e.target.value })} />
            </div>
            <div>
              <label className="label">Business Type</label>
              <select className="select" value={companyForm.business_type || ''} onChange={(e) => setCompanyForm({ ...companyForm, business_type: e.target.value })}>
                <option value="">Select type</option>
                <option value="sole_proprietorship">Sole Proprietorship</option>
                <option value="partnership">Partnership</option>
                <option value="limited_liability">Limited Liability Company</option>
                <option value="ngo">NGO / Non-Profit</option>
              </select>
            </div>
            <div className="col-span-2">
              <label className="label">Address</label>
              <input className="input" value={companyForm.address || ''} onChange={(e) => setCompanyForm({ ...companyForm, address: e.target.value })} />
            </div>
            <div>
              <label className="label">City</label>
              <input className="input" value={companyForm.city || ''} onChange={(e) => setCompanyForm({ ...companyForm, city: e.target.value })} />
            </div>
            <div>
              <label className="label">Region</label>
              <select className="select" value={companyForm.region || ''} onChange={(e) => setCompanyForm({ ...companyForm, region: e.target.value })}>
                <option value="">Select region</option>
                {ghanaRegions.map((r) => <option key={r} value={r}>{r}</option>)}
              </select>
            </div>
            <div>
              <label className="label">Phone</label>
              <input className="input" value={companyForm.phone || ''} onChange={(e) => setCompanyForm({ ...companyForm, phone: e.target.value })} />
            </div>
            <div>
              <label className="label">Email</label>
              <input className="input" type="email" value={companyForm.email || ''} onChange={(e) => setCompanyForm({ ...companyForm, email: e.target.value })} />
            </div>
            <div>
              <label className="label">Financial Year Start</label>
              <select className="select" value={companyForm.financial_year_start || '1'} onChange={(e) => setCompanyForm({ ...companyForm, financial_year_start: e.target.value })}>
                {Array.from({ length: 12 }, (_, i) => (
                  <option key={i + 1} value={i + 1}>{new Date(2024, i).toLocaleString('default', { month: 'long' })}</option>
                ))}
              </select>
            </div>
            <div>
              <label className="label">Base Currency</label>
              <select className="select" value={companyForm.currency || 'GHS'} onChange={(e) => setCompanyForm({ ...companyForm, currency: e.target.value })}>
                <optgroup label="West Africa">
                  <option value="GHS">GHS — Ghana Cedi</option>
                  <option value="NGN">NGN — Nigerian Naira</option>
                  <option value="XOF">XOF — CFA Franc (BCEAO)</option>
                  <option value="GMD">GMD — Gambian Dalasi</option>
                  <option value="SLL">SLL — Sierra Leonean Leone</option>
                  <option value="LRD">LRD — Liberian Dollar</option>
                </optgroup>
                <optgroup label="Major Currencies">
                  <option value="USD">USD — US Dollar</option>
                  <option value="EUR">EUR — Euro</option>
                  <option value="GBP">GBP — British Pound</option>
                </optgroup>
                <optgroup label="Africa">
                  <option value="KES">KES — Kenyan Shilling</option>
                  <option value="ZAR">ZAR — South African Rand</option>
                  <option value="TZS">TZS — Tanzanian Shilling</option>
                  <option value="UGX">UGX — Ugandan Shilling</option>
                  <option value="RWF">RWF — Rwandan Franc</option>
                  <option value="XAF">XAF — CFA Franc (BEAC)</option>
                  <option value="EGP">EGP — Egyptian Pound</option>
                  <option value="MAD">MAD — Moroccan Dirham</option>
                </optgroup>
                <optgroup label="Other">
                  <option value="CNY">CNY — Chinese Yuan</option>
                  <option value="JPY">JPY — Japanese Yen</option>
                  <option value="INR">INR — Indian Rupee</option>
                  <option value="AED">AED — UAE Dirham</option>
                  <option value="CAD">CAD — Canadian Dollar</option>
                  <option value="AUD">AUD — Australian Dollar</option>
                </optgroup>
              </select>
            </div>
            <div>
              <label className="label">Additional Currencies</label>
              <input className="input" placeholder="e.g. USD, EUR, GBP"
                value={companyForm.supported_currencies || ''}
                onChange={(e) => setCompanyForm({ ...companyForm, supported_currencies: e.target.value })}
              />
              <p className="text-xs text-gray-400 mt-1">Comma-separated codes for multi-currency transactions</p>
            </div>
            <div className="col-span-2 border-t border-neutral-200 pt-4 mt-2">
              <label className="label">Default Profit Margin (%)</label>
              <div className="flex items-center gap-3">
                <input
                  className="input w-32"
                  type="number"
                  min="0"
                  max="100"
                  step="0.01"
                  value={companyForm.default_profit_margin ?? '0'}
                  onChange={(e) => setCompanyForm({ ...companyForm, default_profit_margin: e.target.value })}
                />
                <span className="text-sm text-gray-500">Applied as the default markup on cost price for new products</span>
              </div>
            </div>
          </div>
          <div className="mt-6">
            <button className="btn-primary" onClick={saveCompany}>Save Changes</button>
          </div>
        </div>
      )}

      {/* Users Management */}
      {!loading && tab === 'users' && (
        <div>
          <div className="flex justify-end mb-4">
            <button className="btn-primary" onClick={() => {
              setEditUserId(null);
              setUserForm({ role: 'staff' });
              setShowUserModal(true);
            }}>+ Add User</button>
          </div>
          <div className="card">
            <div className="table-container">
              <table className="table">
                <thead>
                  <tr>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Role</th>
                    <th>Status</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {users.map((u: any) => (
                    <tr key={u.id}>
                      <td className="font-medium">{u.first_name} {u.last_name}</td>
                      <td>{u.email}</td>
                      <td><StatusBadge status={u.role} /></td>
                      <td><StatusBadge status={u.is_active ? 'active' : 'inactive'} /></td>
                      <td>
                        <div className="flex gap-2">
                          <button className="text-neutral-900 text-sm" onClick={() => {
                            setEditUserId(u.id);
                            setUserForm({ first_name: u.first_name, last_name: u.last_name, email: u.email, role: u.role, phone: u.phone });
                            setShowUserModal(true);
                          }}>Edit</button>
                          {u.is_active && u.id !== user?.id && (
                            <button className="text-red-600 text-sm" onClick={() => setConfirmDeactivate(u.id)}>Deactivate</button>
                          )}
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          <Modal isOpen={showUserModal} onClose={() => setShowUserModal(false)} title={editUserId ? 'Edit User' : 'Add User'}>
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="label">First Name</label>
                  <input className="input" value={userForm.first_name || ''} onChange={(e) => setUserForm({ ...userForm, first_name: e.target.value })} />
                </div>
                <div>
                  <label className="label">Last Name</label>
                  <input className="input" value={userForm.last_name || ''} onChange={(e) => setUserForm({ ...userForm, last_name: e.target.value })} />
                </div>
              </div>
              <div>
                <label className="label">Email</label>
                <input className="input" type="email" value={userForm.email || ''} onChange={(e) => setUserForm({ ...userForm, email: e.target.value })} />
              </div>
              <div>
                <label className="label">Phone</label>
                <input className="input" value={userForm.phone || ''} onChange={(e) => setUserForm({ ...userForm, phone: e.target.value })} />
              </div>
              {!editUserId && (
                <div>
                  <label className="label">Password</label>
                  <input className="input" type="password" value={userForm.password || ''} onChange={(e) => setUserForm({ ...userForm, password: e.target.value })} />
                </div>
              )}
              <div>
                <label className="label">Role</label>
                <select className="select" value={userForm.role || 'staff'} onChange={(e) => setUserForm({ ...userForm, role: e.target.value })}>
                  <option value="admin">Admin</option>
                  <option value="accountant">Accountant</option>
                  <option value="cashier">Cashier</option>
                  <option value="staff">Staff</option>
                </select>
              </div>
              <div className="flex justify-end gap-3 pt-4">
                <button className="btn-secondary" onClick={() => setShowUserModal(false)}>Cancel</button>
                <button className="btn-primary" onClick={saveUser}>Save</button>
              </div>
            </div>
          </Modal>

          <ConfirmDialog isOpen={!!confirmDeactivate} onClose={() => setConfirmDeactivate(null)}
            title="Deactivate User" message="This user will no longer be able to log in. Continue?"
            onConfirm={deactivateUser} confirmLabel="Deactivate" variant="danger" />
        </div>
      )}

      {/* Bank Accounts */}
      {!loading && tab === 'bank' && (
        <div>
          <div className="flex justify-end mb-4">
            <button className="btn-primary" onClick={() => { setBankForm({}); setShowBankModal(true); }}>+ Add Bank Account</button>
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {bankAccounts.map((b: any) => (
              <div key={b.id} className="card p-5">
                <h4 className="font-semibold text-lg">{b.bank_name}</h4>
                <p className="text-sm text-gray-500">{b.account_name}</p>
                <p className="text-sm text-gray-400">{b.account_number} &middot; {b.branch}</p>
                <p className="text-2xl font-bold text-neutral-900 mt-3">{b.currency || 'GHS'} {parseFloat(b.current_balance || 0).toLocaleString('en-GH', { minimumFractionDigits: 2 })}</p>
                <StatusBadge status={b.is_active ? 'active' : 'inactive'} />
              </div>
            ))}
          </div>

          <Modal isOpen={showBankModal} onClose={() => setShowBankModal(false)} title="Add Bank Account">
            <div className="space-y-4">
              <div>
                <label className="label">Bank Name</label>
                <input className="input" value={bankForm.bank_name || ''} onChange={(e) => setBankForm({ ...bankForm, bank_name: e.target.value })} placeholder="e.g. GCB Bank" />
              </div>
              <div>
                <label className="label">Account Name</label>
                <input className="input" value={bankForm.account_name || ''} onChange={(e) => setBankForm({ ...bankForm, account_name: e.target.value })} />
              </div>
              <div>
                <label className="label">Account Number</label>
                <input className="input" value={bankForm.account_number || ''} onChange={(e) => setBankForm({ ...bankForm, account_number: e.target.value })} />
              </div>
              <div>
                <label className="label">Branch</label>
                <input className="input" value={bankForm.branch || ''} onChange={(e) => setBankForm({ ...bankForm, branch: e.target.value })} />
              </div>
              <div>
                <label className="label">Account Type</label>
                <select className="select" value={bankForm.account_type || ''} onChange={(e) => setBankForm({ ...bankForm, account_type: e.target.value })}>
                  <option value="">Select type</option>
                  <option value="current">Current Account</option>
                  <option value="savings">Savings Account</option>
                  <option value="mobile_money">Mobile Money</option>
                </select>
              </div>
              <div>
                <label className="label">Opening Balance (GHS)</label>
                <input className="input" type="number" step="0.01" value={bankForm.opening_balance || ''} onChange={(e) => setBankForm({ ...bankForm, opening_balance: e.target.value })} />
              </div>
              <div className="flex justify-end gap-3 pt-4">
                <button className="btn-secondary" onClick={() => setShowBankModal(false)}>Cancel</button>
                <button className="btn-primary" onClick={saveBankAccount}>Save</button>
              </div>
            </div>
          </Modal>
        </div>
      )}

      {/* Notifications */}
      {!loading && tab === 'notifications' && (
        <div className="space-y-3 max-w-2xl">
          {notifications.length === 0 && (
            <p className="text-center text-gray-400 py-12">No notifications</p>
          )}
          {notifications.map((n: any) => (
            <div key={n.id} className={`card p-4 flex items-start gap-3 ${!n.is_read ? 'border-l-4 border-neutral-900' : 'opacity-75'}`}>
              <div className="flex-1">
                <p className="font-medium text-sm">{n.title}</p>
                <p className="text-sm text-gray-500">{n.message}</p>
                <p className="text-xs text-gray-400 mt-1">{n.created_at}</p>
              </div>
              {!n.is_read && (
                <button className="text-xs text-neutral-900" onClick={() => markRead(n.id)}>Mark Read</button>
              )}
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
