'use client';

import React, { useState } from 'react';
import { reportAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading } from '@/components/ui';
import FileAttachments from '@/components/ui/FileAttachments';
import DatePicker from '@/components/ui/DatePicker';
import toast from 'react-hot-toast';
import { ArrowDownTrayIcon } from '@heroicons/react/24/outline';
import { useAuth } from '@/context/AuthContext';

export default function ReportsPage() {
  const { user } = useAuth();
  const [tab, setTab] = useState('income');
  const [report, setReport] = useState<any>(null);
  const [loading, setLoading] = useState(false);
  const [dateFrom, setDateFrom] = useState(new Date().getFullYear() + '-01-01');
  const [dateTo, setDateTo] = useState(new Date().toISOString().split('T')[0]);

  const loadReport = async (type: string) => {
    setLoading(true);
    setReport(null);
    try {
      const params = { date_from: dateFrom, date_to: dateTo };
      let res;
      switch (type) {
        case 'income': res = await reportAPI.incomeStatement(params); break;
        case 'balance': res = await reportAPI.balanceSheet(params); break;
        case 'cashflow': res = await reportAPI.cashFlow(params); break;
        case 'tax': res = await reportAPI.taxSummary(params); break;
        case 'aging': res = await reportAPI.arAging(); break;
        default: break;
      }
      setReport(res?.data?.data);
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error loading report');
    }
    setLoading(false);
  };

  const getReportTitle = () => {
    const t = tabs.find(t => t.key === tab);
    return t?.label || 'Report';
  };

  const getReportRows = (): { label: string; value: string }[] => {
    if (!report) return [];
    const rows: { label: string; value: string }[] = [];

    if (tab === 'income') {
      rows.push({ label: '--- REVENUE ---', value: '' });
      report.revenue?.forEach((r: any) => rows.push({ label: r.account_name, value: r.balance?.toString() }));
      rows.push({ label: 'Total Revenue', value: report.total_revenue?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: '--- EXPENSES ---', value: '' });
      report.expenses?.forEach((e: any) => rows.push({ label: e.account_name, value: e.balance?.toString() }));
      rows.push({ label: 'Total Expenses', value: report.total_expenses?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: 'NET PROFIT / (LOSS)', value: report.net_profit?.toString() });
    } else if (tab === 'balance') {
      rows.push({ label: '--- ASSETS ---', value: '' });
      rows.push({ label: 'Current Assets', value: '' });
      report.assets?.current_assets?.forEach((a: any) => rows.push({ label: `    ${a.account_name}`, value: a.balance?.toString() }));
      rows.push({ label: 'Total Current Assets', value: report.assets?.total_current_assets?.toString() });
      rows.push({ label: 'Fixed Assets', value: '' });
      report.assets?.fixed_assets?.forEach((a: any) => rows.push({ label: `    ${a.account_name}`, value: a.balance?.toString() }));
      rows.push({ label: 'Total Fixed Assets', value: report.assets?.total_fixed_assets?.toString() });
      rows.push({ label: 'TOTAL ASSETS', value: report.assets?.total_assets?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: '--- LIABILITIES ---', value: '' });
      rows.push({ label: 'Current Liabilities', value: '' });
      report.liabilities?.current_liabilities?.forEach((l: any) => rows.push({ label: `    ${l.account_name}`, value: l.balance?.toString() }));
      rows.push({ label: 'Total Current Liabilities', value: report.liabilities?.total_current_liabilities?.toString() });
      rows.push({ label: 'Long-Term Liabilities', value: '' });
      report.liabilities?.long_term_liabilities?.forEach((l: any) => rows.push({ label: `    ${l.account_name}`, value: l.balance?.toString() }));
      rows.push({ label: 'Total Long-Term Liabilities', value: report.liabilities?.total_long_term_liabilities?.toString() });
      rows.push({ label: 'TOTAL LIABILITIES', value: report.liabilities?.total_liabilities?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: '--- EQUITY ---', value: '' });
      report.equity?.accounts?.forEach((e: any) => rows.push({ label: `    ${e.account_name}`, value: e.balance?.toString() }));
      rows.push({ label: '    Retained Earnings', value: report.equity?.retained_earnings?.toString() });
      rows.push({ label: 'TOTAL EQUITY', value: report.equity?.total_equity?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: 'LIABILITIES + EQUITY', value: report.total_liabilities_and_equity?.toString() });
    } else if (tab === 'cashflow') {
      rows.push({ label: '--- OPERATING ACTIVITIES ---', value: '' });
      rows.push({ label: '    Net Income', value: report.operating_activities?.net_income?.toString() });
      rows.push({ label: '    Cash from Customers', value: report.operating_activities?.cash_from_customers?.toString() });
      rows.push({ label: '    Cash from POS Sales', value: report.operating_activities?.cash_from_pos_sales?.toString() });
      rows.push({ label: '    Cash Paid to Suppliers', value: report.operating_activities?.cash_paid_to_suppliers?.toString() });
      rows.push({ label: '    Cash Paid for Payroll', value: report.operating_activities?.cash_paid_for_payroll?.toString() });
      rows.push({ label: 'Net Operating Cash Flow', value: report.operating_activities?.net_operating_cash_flow?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: 'NET CASH FLOW', value: report.summary?.net_cash_flow?.toString() });
    } else if (tab === 'tax') {
      rows.push({ label: '--- VAT / LEVIES ---', value: '' });
      rows.push({ label: '    Output VAT (Sales)', value: report.vat?.output_vat?.toString() });
      rows.push({ label: '    Input VAT (Purchases)', value: report.vat?.input_vat?.toString() });
      rows.push({ label: '    VAT Payable', value: report.vat?.vat_payable?.toString() });
      rows.push({ label: '    NHIL Collected', value: report.vat?.nhil_collected?.toString() });
      rows.push({ label: '    GETFund Levy', value: report.vat?.getfund_collected?.toString() });
      rows.push({ label: '', value: '' });
      rows.push({ label: '--- PAYE / SOCIAL SECURITY ---', value: '' });
      rows.push({ label: '    PAYE Tax', value: report.paye?.total_paye_tax?.toString() });
      rows.push({ label: '    SSNIT Contributions', value: report.paye?.total_ssnit?.toString() });
      rows.push({ label: '    Tier 2 Contributions', value: report.paye?.total_tier2?.toString() });
    } else if (tab === 'aging') {
      const agingData = Array.isArray(report) ? report : [];
      agingData.forEach((r: any) => {
        rows.push({ label: r.contact_name, value: `Current: ${r.current || 0}, 30d: ${r.days_30 || 0}, 60d: ${r.days_60 || 0}, 90d: ${r.days_90 || 0}, 90+: ${r.days_over_90 || 0}, Total: ${r.total || 0}` });
      });
    }
    return rows;
  };

  const exportPDF = async () => {
    try {
      const { default: jsPDF } = await import('jspdf');
      const { default: autoTable } = await import('jspdf-autotable');
      const doc = new jsPDF();
      const title = getReportTitle();
      doc.setFontSize(16);
      doc.text(title, 105, 15, { align: 'center' });
      doc.setFontSize(10);
      doc.text(`${dateFrom} to ${dateTo}`, 105, 22, { align: 'center' });

      const rows = getReportRows();
      const tableRows = rows.map(r => [r.label, r.value]);

      autoTable(doc, {
        startY: 30,
        head: [['Description', 'Amount (GHS)']],
        body: tableRows,
        styles: { fontSize: 9 },
        headStyles: { fillColor: [23, 23, 23] },
        didParseCell: (data: any) => {
          if (data.row.raw?.[0]?.toString().startsWith('---') || data.row.raw?.[0]?.toString().startsWith('Total') || data.row.raw?.[0]?.toString().startsWith('NET')) {
            data.cell.styles.fontStyle = 'bold';
          }
        },
      });

      doc.save(`${title.replace(/\s+/g, '_')}_${dateFrom}_${dateTo}.pdf`);
      toast.success('PDF downloaded');
    } catch (err) {
      toast.error('PDF export failed');
      console.error(err);
    }
  };

  const exportExcel = async () => {
    try {
      const XLSX = await import('xlsx');
      const title = getReportTitle();
      const rows = getReportRows();

      if (tab === 'aging' && Array.isArray(report)) {
        const agingRows = report.map((r: any) => ({
          Customer: r.contact_name,
          Current: r.current || 0,
          '1-30 Days': r.days_30 || 0,
          '31-60 Days': r.days_60 || 0,
          '61-90 Days': r.days_90 || 0,
          '90+ Days': r.days_over_90 || 0,
          Total: r.total || 0,
        }));
        const ws = XLSX.utils.json_to_sheet(agingRows);
        const wb = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(wb, ws, title);
        XLSX.writeFile(wb, `${title.replace(/\s+/g, '_')}_${dateTo}.xlsx`);
      } else {
        const data = rows.map(r => ({ Description: r.label, 'Amount (GHS)': r.value }));
        const ws = XLSX.utils.json_to_sheet(data);
        const wb = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(wb, ws, title);
        XLSX.writeFile(wb, `${title.replace(/\s+/g, '_')}_${dateFrom}_${dateTo}.xlsx`);
      }
      toast.success('Excel downloaded');
    } catch (err) {
      toast.error('Excel export failed');
      console.error(err);
    }
  };

  const tabs = [
    { key: 'income', label: 'Income Statement' },
    { key: 'balance', label: 'Balance Sheet' },
    { key: 'cashflow', label: 'Cash Flow' },
    { key: 'tax', label: 'Tax Summary' },
    { key: 'aging', label: 'AR Aging' },
  ];

  return (
    <div className="space-y-6">
      <h1 className="text-2xl font-bold text-gray-900">Financial Reports</h1>

      {/* Controls */}
      <div className="flex items-end gap-4 flex-wrap">
        <div>
          <label className="label">From</label>
          <DatePicker value={dateFrom} onChange={(v) => setDateFrom(v)} />
        </div>
        <div>
          <label className="label">To</label>
          <DatePicker value={dateTo} onChange={(v) => setDateTo(v)} />
        </div>
        <button className="btn-primary" onClick={() => loadReport(tab)}>Generate Report</button>
        <button className="btn-secondary" onClick={() => window.print()}>Print</button>
        {report && (
          <>
            <button className="btn-secondary flex items-center gap-1" onClick={exportPDF}>
              <ArrowDownTrayIcon className="h-4 w-4" /> PDF
            </button>
            <button className="btn-secondary flex items-center gap-1" onClick={exportExcel}>
              <ArrowDownTrayIcon className="h-4 w-4" /> Excel
            </button>
          </>
        )}
      </div>

      {/* Tabs */}
      <div className="border-b border-gray-200">
        <nav className="flex gap-6">
          {tabs.map((t) => (
            <button key={t.key}
              className={`pb-3 text-sm font-medium border-b-2 transition-colors ${
                tab === t.key ? 'border-neutral-900 text-neutral-900' : 'border-transparent text-gray-500'
              }`}
              onClick={() => { setTab(t.key); setReport(null); }}
            >{t.label}</button>
          ))}
        </nav>
      </div>

      {loading && <PageLoading />}

      {!loading && !report && (
        <div className="text-center py-16 text-gray-400">
          <p className="text-lg">Select date range and click &quot;Generate Report&quot;</p>
        </div>
      )}

      {/* Income Statement */}
      {report && tab === 'income' && (
        <div className="card p-8 max-w-3xl mx-auto">
          <h2 className="text-xl font-bold text-center mb-1">Income Statement (Profit & Loss)</h2>
          <p className="text-sm text-gray-500 text-center mb-8">{dateFrom} to {dateTo}</p>

          <h3 className="font-semibold text-gray-700 mb-2 border-b pb-1">Revenue</h3>
          {report.revenue?.map((r: any) => (
            <div key={r.account_name} className="flex justify-between py-1.5 text-sm">
              <span className="text-gray-600 pl-4">{r.account_name}</span>
              <span>{formatCurrency(r.balance)}</span>
            </div>
          ))}
          <div className="flex justify-between py-2 font-bold border-t mt-2">
            <span>Total Revenue</span><span>{formatCurrency(report.total_revenue)}</span>
          </div>

          <h3 className="font-semibold text-gray-700 mb-2 border-b pb-1 mt-6">Expenses</h3>
          {report.expenses?.map((e: any) => (
            <div key={e.account_name} className="flex justify-between py-1.5 text-sm">
              <span className="text-gray-600 pl-4">{e.account_name}</span>
              <span>{formatCurrency(e.balance)}</span>
            </div>
          ))}
          <div className="flex justify-between py-2 font-bold border-t mt-2">
            <span>Total Expenses</span><span>{formatCurrency(report.total_expenses)}</span>
          </div>

          <div className={`flex justify-between py-3 mt-4 text-xl font-bold border-t-2 ${
            parseFloat(report.net_profit) >= 0 ? 'text-green-600' : 'text-red-600'
          }`}>
            <span>Net Profit / (Loss)</span>
            <span>{formatCurrency(report.net_profit)}</span>
          </div>
        </div>
      )}

      {/* Balance Sheet */}
      {report && tab === 'balance' && (
        <div className="card p-8 max-w-3xl mx-auto">
          <h2 className="text-xl font-bold text-center mb-1">Balance Sheet</h2>
          <p className="text-sm text-gray-500 text-center mb-8">As at {dateTo}</p>

          {/* ASSETS */}
          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3">Assets</h3>

          <h4 className="font-semibold text-gray-600 mb-1 ml-2">Current Assets</h4>
          {report.assets?.current_assets?.map((a: any) => (
            <div key={a.account_name} className="flex justify-between py-1 text-sm">
              <span className="text-gray-600 pl-6">{a.account_name}</span>
              <span>{formatCurrency(a.balance)}</span>
            </div>
          ))}
          {(!report.assets?.current_assets || report.assets.current_assets.length === 0) && (
            <p className="text-sm text-gray-400 pl-6 py-1">No current assets</p>
          )}
          <div className="flex justify-between py-1.5 font-semibold text-sm border-t mt-1 ml-2">
            <span>Total Current Assets</span><span>{formatCurrency(report.assets?.total_current_assets || 0)}</span>
          </div>

          <h4 className="font-semibold text-gray-600 mb-1 ml-2 mt-4">Fixed Assets</h4>
          {report.assets?.fixed_assets?.map((a: any) => (
            <div key={a.account_name} className="flex justify-between py-1 text-sm">
              <span className="text-gray-600 pl-6">{a.account_name}</span>
              <span>{formatCurrency(a.balance)}</span>
            </div>
          ))}
          {(!report.assets?.fixed_assets || report.assets.fixed_assets.length === 0) && (
            <p className="text-sm text-gray-400 pl-6 py-1">No fixed assets</p>
          )}
          <div className="flex justify-between py-1.5 font-semibold text-sm border-t mt-1 ml-2">
            <span>Total Fixed Assets</span><span>{formatCurrency(report.assets?.total_fixed_assets || 0)}</span>
          </div>

          <div className="flex justify-between py-2 font-bold border-t-2 mt-3 text-base">
            <span>Total Assets</span><span>{formatCurrency(report.assets?.total_assets || 0)}</span>
          </div>

          {/* LIABILITIES */}
          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3 mt-8">Liabilities</h3>

          <h4 className="font-semibold text-gray-600 mb-1 ml-2">Current Liabilities</h4>
          {report.liabilities?.current_liabilities?.map((l: any) => (
            <div key={l.account_name} className="flex justify-between py-1 text-sm">
              <span className="text-gray-600 pl-6">{l.account_name}</span>
              <span>{formatCurrency(l.balance)}</span>
            </div>
          ))}
          {(!report.liabilities?.current_liabilities || report.liabilities.current_liabilities.length === 0) && (
            <p className="text-sm text-gray-400 pl-6 py-1">No current liabilities</p>
          )}
          <div className="flex justify-between py-1.5 font-semibold text-sm border-t mt-1 ml-2">
            <span>Total Current Liabilities</span><span>{formatCurrency(report.liabilities?.total_current_liabilities || 0)}</span>
          </div>

          <h4 className="font-semibold text-gray-600 mb-1 ml-2 mt-4">Long-Term Liabilities</h4>
          {report.liabilities?.long_term_liabilities?.map((l: any) => (
            <div key={l.account_name} className="flex justify-between py-1 text-sm">
              <span className="text-gray-600 pl-6">{l.account_name}</span>
              <span>{formatCurrency(l.balance)}</span>
            </div>
          ))}
          {(!report.liabilities?.long_term_liabilities || report.liabilities.long_term_liabilities.length === 0) && (
            <p className="text-sm text-gray-400 pl-6 py-1">No long-term liabilities</p>
          )}
          <div className="flex justify-between py-1.5 font-semibold text-sm border-t mt-1 ml-2">
            <span>Total Long-Term Liabilities</span><span>{formatCurrency(report.liabilities?.total_long_term_liabilities || 0)}</span>
          </div>

          <div className="flex justify-between py-2 font-bold border-t-2 mt-3 text-base">
            <span>Total Liabilities</span><span>{formatCurrency(report.liabilities?.total_liabilities || 0)}</span>
          </div>

          {/* EQUITY */}
          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3 mt-8">Equity</h3>
          {report.equity?.accounts?.map((e: any) => (
            <div key={e.account_name} className="flex justify-between py-1 text-sm">
              <span className="text-gray-600 pl-4">{e.account_name}</span>
              <span>{formatCurrency(e.balance)}</span>
            </div>
          ))}
          <div className="flex justify-between py-1 text-sm">
            <span className="text-gray-600 pl-4">Retained Earnings (YTD)</span>
            <span>{formatCurrency(report.equity?.retained_earnings || 0)}</span>
          </div>
          <div className="flex justify-between py-2 font-bold border-t-2 mt-3 text-base">
            <span>Total Equity</span><span>{formatCurrency(report.equity?.total_equity || 0)}</span>
          </div>

          {/* Balance Check */}
          <div className={`flex justify-between py-3 mt-6 text-lg font-bold border-t-4 border-double ${
            report.is_balanced ? 'text-green-700 border-green-300' : 'text-red-600 border-red-300'
          }`}>
            <span>Liabilities + Equity</span>
            <span>{formatCurrency(report.total_liabilities_and_equity || 0)}</span>
          </div>
          {report.is_balanced ? (
            <p className="text-center text-green-600 text-sm mt-2 font-medium">&#10003; Balance sheet is balanced</p>
          ) : (
            <p className="text-center text-red-600 text-sm mt-2 font-medium">&#10007; Balance sheet is NOT balanced — difference: {
              formatCurrency(Math.abs((report.assets?.total_assets || 0) - (report.total_liabilities_and_equity || 0)))
            }</p>
          )}
        </div>
      )}

      {/* Cash Flow */}
      {report && tab === 'cashflow' && (
        <div className="card p-8 max-w-3xl mx-auto">
          <h2 className="text-xl font-bold text-center mb-1">Cash Flow Statement</h2>
          <p className="text-sm text-gray-500 text-center mb-8">{dateFrom} to {dateTo}</p>

          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3">Operating Activities</h3>
          <div className="flex justify-between py-1.5 text-sm">
            <span className="text-gray-600 pl-4">Net Income</span>
            <span>{formatCurrency(report.operating_activities?.net_income || 0)}</span>
          </div>
          <div className="flex justify-between py-1.5 text-sm">
            <span className="text-gray-600 pl-4">Cash from Customers</span>
            <span className="text-green-600">{formatCurrency(report.operating_activities?.cash_from_customers || 0)}</span>
          </div>
          <div className="flex justify-between py-1.5 text-sm">
            <span className="text-gray-600 pl-4">Cash from POS Sales</span>
            <span className="text-green-600">{formatCurrency(report.operating_activities?.cash_from_pos_sales || 0)}</span>
          </div>
          <div className="flex justify-between py-1.5 text-sm">
            <span className="text-gray-600 pl-4">Cash Paid to Suppliers</span>
            <span className="text-red-500">({formatCurrency(report.operating_activities?.cash_paid_to_suppliers || 0)})</span>
          </div>
          <div className="flex justify-between py-1.5 text-sm">
            <span className="text-gray-600 pl-4">Cash Paid for Payroll</span>
            <span className="text-red-500">({formatCurrency(report.operating_activities?.cash_paid_for_payroll || 0)})</span>
          </div>
          <div className="flex justify-between py-2 font-bold border-t mt-2">
            <span>Net Operating Cash Flow</span>
            <span className={parseFloat(report.operating_activities?.net_operating_cash_flow || 0) >= 0 ? 'text-green-600' : 'text-red-600'}>
              {formatCurrency(report.operating_activities?.net_operating_cash_flow || 0)}
            </span>
          </div>

          <div className={`flex justify-between py-3 mt-6 text-lg font-bold border-t-4 border-double ${
            parseFloat(report.summary?.net_cash_flow || 0) >= 0 ? 'text-green-700 border-green-300' : 'text-red-600 border-red-300'
          }`}>
            <span>Net Cash Flow</span>
            <span>{formatCurrency(report.summary?.net_cash_flow || 0)}</span>
          </div>
        </div>
      )}

      {/* Tax Summary */}
      {report && tab === 'tax' && (
        <div className="card p-8 max-w-3xl mx-auto">
          <h2 className="text-xl font-bold text-center mb-1">Tax Summary</h2>
          <p className="text-sm text-gray-500 text-center mb-8">{dateFrom} to {dateTo}</p>

          {/* VAT / Levies Section */}
          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3">VAT &amp; Levies</h3>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">Output VAT (Sales)</span>
            <span>{formatCurrency(report.vat?.output_vat || 0)}</span>
          </div>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">Input VAT (Purchases)</span>
            <span className="text-red-500">({formatCurrency(report.vat?.input_vat || 0)})</span>
          </div>
          <div className="flex justify-between py-2 font-bold border-t mt-1">
            <span className="pl-4">VAT Payable to GRA</span>
            <span className={parseFloat(report.vat?.vat_payable || 0) >= 0 ? 'text-red-600' : 'text-green-600'}>
              {formatCurrency(report.vat?.vat_payable || 0)}
            </span>
          </div>
          <div className="flex justify-between py-2 text-sm mt-2">
            <span className="text-gray-600 pl-4">NHIL Collected</span>
            <span>{formatCurrency(report.vat?.nhil_collected || 0)}</span>
          </div>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">GETFund Levy Collected</span>
            <span>{formatCurrency(report.vat?.getfund_collected || 0)}</span>
          </div>
          <div className="flex justify-between py-2 font-bold border-t mt-1">
            <span>Total VAT/Levies Payable</span>
            <span>{formatCurrency(
              (parseFloat(report.vat?.vat_payable) || 0) +
              (parseFloat(report.vat?.nhil_collected) || 0) +
              (parseFloat(report.vat?.getfund_collected) || 0)
            )}</span>
          </div>

          {/* PAYE / Social Security Section */}
          <h3 className="font-bold text-gray-800 text-lg border-b-2 pb-1 mb-3 mt-8">PAYE &amp; Social Security</h3>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">PAYE Tax Deducted</span>
            <span>{formatCurrency(report.paye?.total_paye_tax || 0)}</span>
          </div>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">SSNIT Contributions (Employee + Employer)</span>
            <span>{formatCurrency(report.paye?.total_ssnit || 0)}</span>
          </div>
          <div className="flex justify-between py-2 text-sm">
            <span className="text-gray-600 pl-4">Tier 2 Contributions</span>
            <span>{formatCurrency(report.paye?.total_tier2 || 0)}</span>
          </div>
          <div className="flex justify-between py-2 font-bold border-t mt-1">
            <span>Total PAYE/Social Security</span>
            <span>{formatCurrency(
              (parseFloat(report.paye?.total_paye_tax) || 0) +
              (parseFloat(report.paye?.total_ssnit) || 0) +
              (parseFloat(report.paye?.total_tier2) || 0)
            )}</span>
          </div>

          {/* Grand Total */}
          <div className="flex justify-between py-3 mt-6 text-lg font-bold border-t-4 border-double text-red-700">
            <span>Total Tax Obligations</span>
            <span>{formatCurrency(
              (parseFloat(report.vat?.vat_payable) || 0) +
              (parseFloat(report.vat?.nhil_collected) || 0) +
              (parseFloat(report.vat?.getfund_collected) || 0) +
              (parseFloat(report.paye?.total_paye_tax) || 0) +
              (parseFloat(report.paye?.total_ssnit) || 0) +
              (parseFloat(report.paye?.total_tier2) || 0)
            )}</span>
          </div>
        </div>
      )}

      {/* AR Aging */}
      {report && tab === 'aging' && (
        <div className="card">
          <div className="card-header">
            <h3 className="font-semibold">Accounts Receivable Aging</h3>
          </div>
          <div className="table-container">
            <table className="table">
              <thead>
                <tr>
                  <th>Customer</th>
                  <th className="text-right">Current</th>
                  <th className="text-right">1-30 Days</th>
                  <th className="text-right">31-60 Days</th>
                  <th className="text-right">61-90 Days</th>
                  <th className="text-right">90+ Days</th>
                  <th className="text-right">Total</th>
                </tr>
              </thead>
              <tbody>
                {(Array.isArray(report) ? report : []).map((r: any) => (
                  <tr key={r.contact_name}>
                    <td className="font-medium">{r.contact_name}</td>
                    <td className="text-right">{formatCurrency(r.current || 0)}</td>
                    <td className="text-right">{formatCurrency(r.days_30 || 0)}</td>
                    <td className="text-right">{formatCurrency(r.days_60 || 0)}</td>
                    <td className="text-right">{formatCurrency(r.days_90 || 0)}</td>
                    <td className="text-right text-red-600">{formatCurrency(r.days_over_90 || 0)}</td>
                    <td className="text-right font-bold">{formatCurrency(r.total || 0)}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Report Documents */}
      {user?.company?.id && (
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Report Documents</h3>
            <p className="text-xs text-gray-500 mt-1">Attach supporting documents for reports (bank statements, receipts, audit papers, etc.)</p>
          </div>
          <div className="card-body">
            <FileAttachments entityType="report" entityId={user.company.id} />
          </div>
        </div>
      )}
    </div>
  );
}
