'use client';

import React, { useState, useEffect } from 'react';
import { reportAPI } from '@/lib/api';
import { formatCurrency } from '@/lib/utils';
import { StatCard, PageLoading } from '@/components/ui';
import {
  DocumentTextIcon,
  ShoppingCartIcon,
  BanknotesIcon,
  ArrowTrendingUpIcon,
  ArrowTrendingDownIcon,
  CurrencyDollarIcon,
} from '@heroicons/react/24/outline';
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  BarElement,
  LineElement,
  PointElement,
  Title,
  Tooltip,
  Legend,
  ArcElement,
} from 'chart.js';
import { Bar, Doughnut } from 'react-chartjs-2';

ChartJS.register(
  CategoryScale, LinearScale, BarElement, LineElement, PointElement, Title, Tooltip, Legend, ArcElement
);

export default function DashboardPage() {
  const [data, setData] = useState<any>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const load = async () => {
      try {
        const { data: res } = await reportAPI.dashboard();
        setData(res.data);
      } catch (err) {
        console.error('Dashboard load error:', err);
      }
      setLoading(false);
    };
    load();
  }, []);

  if (loading) return <PageLoading />;

  // API returns flat keys (monthly_revenue, monthly_trend, etc.) — map them
  const kpi = data || {};
  const trends = data?.monthly_trend || [];
  const posSummary = data?.pos_summary || {};
  const posDailyTrend = data?.pos_daily_trend || [];
  const cashDist = data?.cash_distribution || [];

  // Revenue chart
  const revenueChart = {
    labels: trends.map((t: any) => t.month),
    datasets: [
      {
        label: 'Revenue',
        data: trends.map((t: any) => parseFloat(t.revenue || 0)),
        backgroundColor: '#171717',
        borderRadius: 4,
      },
      {
        label: 'Expenses',
        data: trends.map((t: any) => parseFloat(t.expenses || 0)),
        backgroundColor: '#d4d4d4',
        borderRadius: 4,
      },
    ],
  };

  // Cash breakdown
  const cashBreakdown = {
    labels: ['Revenue', 'Expenses', 'Profit'],
    datasets: [
      {
        data: [
          parseFloat(kpi.monthly_revenue || 0),
          parseFloat(kpi.monthly_expenses || 0),
          parseFloat(kpi.monthly_profit || 0),
        ],
        backgroundColor: ['#171717', '#737373', '#10b981'],
        borderWidth: 0,
      },
    ],
  };

  // Cash Distribution chart (momo, cash, bank, etc.)
  const distributionColors = ['#171717', '#10b981', '#3b82f6', '#f59e0b', '#ef4444', '#8b5cf6'];
  const cashDistChart = {
    labels: cashDist.map((d: any) => d.method),
    datasets: [
      {
        data: cashDist.map((d: any) => parseFloat(d.total || 0)),
        backgroundColor: cashDist.map((_: any, i: number) => distributionColors[i % distributionColors.length]),
        borderWidth: 0,
      },
    ],
  };

  // POS daily trend chart
  const posDailyChart = {
    labels: posDailyTrend.map((d: any) => {
      const date = new Date(d.sale_day);
      return date.toLocaleDateString('en-GB', { weekday: 'short', day: 'numeric' });
    }),
    datasets: [
      {
        label: 'POS Revenue',
        data: posDailyTrend.map((d: any) => parseFloat(d.total || 0)),
        backgroundColor: '#10b981',
        borderRadius: 4,
      },
    ],
  };

  return (
    <div className="space-y-8">
      {/* Header */}
      <div>
        <h1 className="text-2xl font-bold text-gray-900">Dashboard</h1>
        <p className="text-sm text-gray-500 mt-1">Overview of your business performance</p>
      </div>

      {/* KPI Cards */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
        <StatCard
          title="Total Revenue"
          value={formatCurrency(kpi.monthly_revenue || 0)}
          subtitle="This month"
          icon={<ArrowTrendingUpIcon className="h-5 w-5" />}
        />
        <StatCard
          title="Total Expenses"
          value={formatCurrency(kpi.monthly_expenses || 0)}
          subtitle="This month"
          icon={<ArrowTrendingDownIcon className="h-5 w-5" />}
        />
        <StatCard
          title="Outstanding Invoices"
          value={formatCurrency(kpi.total_receivables || 0)}
          subtitle={`${kpi.overdue_invoices || 0} overdue`}
          icon={<DocumentTextIcon className="h-5 w-5" />}
        />
        <StatCard
          title="POS Sales Today"
          value={formatCurrency(kpi.today_pos_revenue || 0)}
          subtitle={`${kpi.today_pos_sales || 0} transactions`}
          icon={<ShoppingCartIcon className="h-5 w-5" />}
        />
      </div>

      {/* Charts Row 1 */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Revenue & Expense Trend */}
        <div className="card col-span-2">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Revenue vs Expenses (6 Months)</h3>
          </div>
          <div className="card-body">
            <Bar
              data={revenueChart}
              options={{
                responsive: true,
                plugins: { legend: { position: 'bottom' } },
                scales: {
                  y: {
                    beginAtZero: true,
                    ticks: {
                      callback: (value) => `GHS ${Number(value).toLocaleString()}`,
                    },
                  },
                },
              }}
              height={80}
            />
          </div>
        </div>

        {/* Cash Breakdown */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Financial Summary</h3>
          </div>
          <div className="card-body flex items-center justify-center">
            <div className="w-48">
              <Doughnut
                data={cashBreakdown}
                options={{
                  responsive: true,
                  plugins: { legend: { position: 'bottom', labels: { boxWidth: 12 } } },
                  cutout: '65%',
                }}
              />
            </div>
          </div>
        </div>
      </div>

      {/* POS Sales & Cash Distribution Row */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* POS Sales Summary */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900 flex items-center gap-2">
              <ShoppingCartIcon className="h-4 w-4" /> POS Sales Summary
            </h3>
            <p className="text-xs text-gray-400 mt-0.5">This month</p>
          </div>
          <div className="card-body">
            <div className="space-y-4">
              {[
                { label: 'Total Sales', value: posSummary.month_sales || 0, format: false },
                { label: 'Total Revenue', value: posSummary.month_revenue || 0, format: true },
                { label: 'Average Sale', value: posSummary.avg_sale || 0, format: true },
                { label: 'Largest Sale', value: posSummary.largest_sale || 0, format: true },
                { label: 'Today\'s Sales', value: kpi.today_pos_sales || 0, format: false },
                { label: 'Today\'s Revenue', value: kpi.today_pos_revenue || 0, format: true },
              ].map((item) => (
                <div key={item.label} className="flex items-center justify-between py-2 border-b border-gray-100 last:border-0">
                  <span className="text-sm text-gray-600">{item.label}</span>
                  <span className="text-sm font-semibold text-gray-900">
                    {item.format ? formatCurrency(item.value) : item.value}
                  </span>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* POS Daily Trend */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">POS Daily Trend (7 Days)</h3>
          </div>
          <div className="card-body">
            {posDailyTrend.length > 0 ? (
              <Bar
                data={posDailyChart}
                options={{
                  responsive: true,
                  plugins: { legend: { display: false } },
                  scales: {
                    y: {
                      beginAtZero: true,
                      ticks: {
                        callback: (value) => `GHS ${Number(value).toLocaleString()}`,
                      },
                    },
                  },
                }}
                height={160}
              />
            ) : (
              <p className="text-sm text-gray-400 text-center py-8">No POS sales in the last 7 days</p>
            )}
          </div>
        </div>

        {/* Cash Distribution */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900 flex items-center gap-2">
              <BanknotesIcon className="h-4 w-4" /> Cash Distribution
            </h3>
            <p className="text-xs text-gray-400 mt-0.5">Payment methods this month</p>
          </div>
          <div className="card-body">
            {cashDist.length > 0 ? (
              <div className="space-y-4">
                <div className="flex items-center justify-center">
                  <div className="w-44">
                    <Doughnut
                      data={cashDistChart}
                      options={{
                        responsive: true,
                        plugins: { legend: { display: false } },
                        cutout: '60%',
                      }}
                    />
                  </div>
                </div>
                <div className="space-y-2 pt-2">
                  {cashDist.map((d: any, i: number) => (
                    <div key={d.method} className="flex items-center justify-between text-sm">
                      <div className="flex items-center gap-2">
                        <span className="w-3 h-3 rounded-full flex-shrink-0" style={{ backgroundColor: distributionColors[i % distributionColors.length] }} />
                        <span className="text-gray-600">{d.method}</span>
                      </div>
                      <div className="text-right">
                        <span className="font-semibold text-gray-900">{formatCurrency(d.total)}</span>
                        <span className="text-xs text-gray-400 ml-1">({d.count})</span>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            ) : (
              <p className="text-sm text-gray-400 text-center py-8">No payment data yet</p>
            )}
          </div>
        </div>
      </div>

      {/* Bottom Row */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Quick Stats */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Quick Stats</h3>
          </div>
          <div className="card-body">
            <div className="space-y-4">
              {[
                { label: 'Total Receivables', value: formatCurrency(kpi.total_receivables || 0) },
                { label: 'Total Payables', value: formatCurrency(kpi.total_payables || 0) },
                { label: 'Overdue Invoices', value: kpi.overdue_invoices || 0 },
                { label: 'Low Stock Alerts', value: kpi.low_stock_alerts || 0 },
              ].map((item) => (
                <div key={item.label} className="flex items-center justify-between py-2 border-b border-gray-100 last:border-0">
                  <span className="text-sm text-gray-600">{item.label}</span>
                  <span className="text-sm font-semibold text-gray-900">{item.value}</span>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Recent Activity */}
        <div className="card">
          <div className="card-header">
            <h3 className="text-sm font-semibold text-gray-900">Recent Activity</h3>
          </div>
          <div className="card-body">
            {data?.recent_activity && data.recent_activity.length > 0 ? (
              <div className="space-y-3">
                {data.recent_activity.slice(0, 8).map((activity: any, i: number) => (
                  <div key={i} className="flex items-center gap-3 text-sm">
                    <div className="w-1.5 h-1.5 rounded-full bg-neutral-400 flex-shrink-0" />
                    <span className="text-gray-700 flex-1">{activity.description}</span>
                    <span className="text-gray-400 text-xs whitespace-nowrap">{activity.time}</span>
                  </div>
                ))}
              </div>
            ) : (
              <p className="text-sm text-gray-400 text-center py-8">No recent activity</p>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
