'use client';

import React, { useState, useEffect } from 'react';
import { assetAPI } from '@/lib/api';
import { formatCurrency, formatDate } from '@/lib/utils';
import { PageLoading, StatusBadge, Modal, EmptyState } from '@/components/ui';
import DatePicker from '@/components/ui/DatePicker';
import toast from 'react-hot-toast';
import { PlusIcon, BuildingOfficeIcon } from '@heroicons/react/24/outline';

export default function AssetsPage() {
  const [assets, setAssets] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);

  const [form, setForm] = useState({
    asset_name: '', category: 'equipment', description: '',
    purchase_date: new Date().toISOString().split('T')[0],
    purchase_cost: '', salvage_value: '0', useful_life_months: '60',
    depreciation_method: 'straight_line',
  });

  useEffect(() => { loadData(); }, []);

  const loadData = async () => {
    try {
      const { data } = await assetAPI.list();
      setAssets(data?.data || []);
    } catch (err) { console.error(err); }
    setLoading(false);
  };

  const createAsset = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await assetAPI.create(form);
      toast.success('Asset created with depreciation schedule');
      setShowModal(false);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  const runDep = async (id: string) => {
    try {
      const { data } = await assetAPI.depreciate(id);
      toast.success(`Depreciation recorded: ${formatCurrency(data?.data?.depreciation_amount || 0)}`);
      loadData();
    } catch (err: any) {
      toast.error(err?.response?.data?.message || 'Error');
    }
  };

  if (loading) return <PageLoading />;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Fixed Assets</h1>
        <button className="btn-primary" onClick={() => setShowModal(true)}>
          <PlusIcon className="h-4 w-4 mr-1" /> Add Asset
        </button>
      </div>

      <div className="card">
        <div className="table-container">
          <table className="table">
            <thead>
              <tr>
                <th>Asset Name</th>
                <th>Category</th>
                <th>Purchase Date</th>
                <th className="text-right">Cost</th>
                <th className="text-right">Book Value</th>
                <th className="text-right">Accum. Dep.</th>
                <th>Status</th>
                <th></th>
              </tr>
            </thead>
            <tbody>
              {assets.length === 0 ? (
                <tr><td colSpan={8}><EmptyState title="No assets" icon={<BuildingOfficeIcon className="h-12 w-12" />} /></td></tr>
              ) : (
                assets.map((a: any) => (
                  <tr key={a.id}>
                    <td className="font-medium">{a.asset_name}</td>
                    <td className="capitalize">{a.category}</td>
                    <td>{formatDate(a.purchase_date)}</td>
                    <td className="text-right">{formatCurrency(a.purchase_cost)}</td>
                    <td className="text-right font-medium">{formatCurrency(a.current_value)}</td>
                    <td className="text-right text-gray-500">{formatCurrency(a.accumulated_depreciation || 0)}</td>
                    <td><StatusBadge status={a.status} /></td>
                    <td>
                      {a.status === 'active' && (
                        <button className="btn-secondary btn-sm" onClick={() => runDep(a.id)}>Depreciate</button>
                      )}
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      <Modal open={showModal} onClose={() => setShowModal(false)} title="Add Fixed Asset" size="lg">
        <form onSubmit={createAsset} className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Asset Name *</label>
              <input className="input" value={form.asset_name} onChange={(e) => setForm({ ...form, asset_name: e.target.value })} required />
            </div>
            <div>
              <label className="label">Category *</label>
              <select className="select" value={form.category} onChange={(e) => setForm({ ...form, category: e.target.value })}>
                <option value="equipment">Equipment</option>
                <option value="vehicle">Vehicle</option>
                <option value="furniture">Furniture & Fittings</option>
                <option value="building">Building</option>
                <option value="land">Land</option>
                <option value="computer">Computer / IT</option>
                <option value="other">Other</option>
              </select>
            </div>
          </div>
          <div>
            <label className="label">Description</label>
            <textarea className="input" rows={2} value={form.description} onChange={(e) => setForm({ ...form, description: e.target.value })} />
          </div>
          <div className="grid grid-cols-3 gap-4">
            <div>
              <label className="label">Purchase Date *</label>
              <DatePicker value={form.purchase_date} onChange={(v) => setForm({ ...form, purchase_date: v })} required />
            </div>
            <div>
              <label className="label">Purchase Cost (GHS) *</label>
              <input type="number" step="0.01" className="input" value={form.purchase_cost} onChange={(e) => setForm({ ...form, purchase_cost: e.target.value })} required />
            </div>
            <div>
              <label className="label">Salvage Value (GHS)</label>
              <input type="number" step="0.01" className="input" value={form.salvage_value} onChange={(e) => setForm({ ...form, salvage_value: e.target.value })} />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="label">Useful Life (months) *</label>
              <input type="number" className="input" value={form.useful_life_months} onChange={(e) => setForm({ ...form, useful_life_months: e.target.value })} required />
            </div>
            <div>
              <label className="label">Depreciation Method</label>
              <select className="select" value={form.depreciation_method} onChange={(e) => setForm({ ...form, depreciation_method: e.target.value })}>
                <option value="straight_line">Straight Line</option>
                <option value="declining_balance">Declining Balance</option>
              </select>
            </div>
          </div>
          <div className="flex justify-end gap-3 pt-2">
            <button type="button" className="btn-secondary" onClick={() => setShowModal(false)}>Cancel</button>
            <button type="submit" className="btn-primary">Create Asset</button>
          </div>
        </form>
      </Modal>
    </div>
  );
}
