#!/bin/bash
# CashBook ERP - cPanel Quick Deployment Script
# 
# This script prepares the application for cPanel upload.
# Usage: bash deploy-cpanel.sh

set -e

echo "=== CashBook ERP - cPanel Deployment Preparation ==="
echo ""

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Step 1: Build frontend static export
echo -e "${BLUE}Step 1: Building frontend static export...${NC}"
cd frontend
npm install --prefer-offline --no-audit
npm run export
echo -e "${GREEN}✓ Frontend build complete. Files in ./out/${NC}"
cd ..

# Step 2: Prepare backend
echo ""
echo -e "${BLUE}Step 2: Preparing backend dependencies...${NC}"
cd backend
composer install --optimize-autoloader --no-dev
echo -e "${GREEN}✓ Backend dependencies installed${NC}"
cd ..

# Step 3: Create deployment package
echo ""
echo -e "${BLUE}Step 3: Creating deployment packages...${NC}"

# Frontend package
echo "Creating frontend package..."
cd frontend/out
zip -r ../../cashbook-frontend.zip . > /dev/null 2>&1 || true
cd ../..
echo -e "${GREEN}✓ Frontend package: cashbook-frontend.zip${NC}"

# Backend package (excluding node_modules, .git, etc.)
echo "Creating backend package..."
cd backend
zip -r ../cashbook-backend.zip \
  public/ src/ config/ routes/ vendor/ \
  .htaccess .env.cpanel.example \
  composer.json composer.lock \
  -x \
  "*.git*" \
  ".DS_Store" \
  "*.log" \
  > /dev/null 2>&1 || true
cd ..
echo -e "${GREEN}✓ Backend package: cashbook-backend.zip${NC}"

# Step 4: Database schema
echo ""
echo -e "${BLUE}Step 4: Preparing database schema...${NC}"
cp backend/database/schema.cpanel.sql . && \
echo -e "${GREEN}✓ Database schema: schema.cpanel.sql${NC}"

# Step 5: Documentation
echo ""
echo -e "${BLUE}Step 5: Checking documentation...${NC}"
if [ -f CPANEL_DEPLOYMENT.md ]; then
  echo -e "${GREEN}✓ Deployment guide: CPANEL_DEPLOYMENT.md${NC}"
else
  echo -e "${RED}✗ Warning: CPANEL_DEPLOYMENT.md not found${NC}"
fi

# Summary
echo ""
echo -e "${GREEN}=== Deployment Preparation Complete ===${NC}"
echo ""
echo "Next steps:"
echo "1. Log into cPanel at https://your-domain.com:2083"
echo "2. Go to File Manager"
echo "3. Upload cashbook-frontend.zip to public_html and extract"
echo "4. Create a folder 'api' in public_html"
echo "5. Upload cashbook-backend.zip to public_html/api and extract"
echo "6. Go to PostgreSQL Databases and create database 'cashbook_prod'"
echo "7. Go to phpPgAdmin and run schema.cpanel.sql"
echo "8. Update .env file in public_html/api/ with database credentials"
echo "9. Test: https://your-domain.com/"
echo ""
echo "See CPANEL_DEPLOYMENT.md for detailed instructions."
echo ""
