<?php

namespace CashBook\Core;

class Router
{
    private array $routes = [];
    private array $groupMiddleware = [];

    public function get(string $path, $handler, array $middleware = []): void
    {
        $this->addRoute('GET', $path, $handler, $middleware);
    }

    public function post(string $path, $handler, array $middleware = []): void
    {
        $this->addRoute('POST', $path, $handler, $middleware);
    }

    public function put(string $path, $handler, array $middleware = []): void
    {
        $this->addRoute('PUT', $path, $handler, $middleware);
    }

    public function patch(string $path, $handler, array $middleware = []): void
    {
        $this->addRoute('PATCH', $path, $handler, $middleware);
    }

    public function delete(string $path, $handler, array $middleware = []): void
    {
        $this->addRoute('DELETE', $path, $handler, $middleware);
    }

    /**
     * Group routes with shared middleware.
     * Usage: $router->group(['middleware' => [...]], function($router) { ... });
     */
    public function group(array $options, callable $callback): void
    {
        $previousMiddleware = $this->groupMiddleware;
        $newMiddleware = $options['middleware'] ?? [];
        $this->groupMiddleware = array_merge($this->groupMiddleware, $newMiddleware);

        $callback($this);

        $this->groupMiddleware = $previousMiddleware;
    }

    public function addRoute(string $method, string $path, $handler, array $middleware = []): void
    {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'handler' => $handler,
            'middleware' => array_merge($this->groupMiddleware, $middleware)
        ];
    }

    public function dispatch(Request $request): void
    {
        foreach ($this->routes as $route) {
            if ($route['method'] !== $request->getMethod()) {
                continue;
            }

            $params = $this->matchRoute($route['path'], $request->getUri());
            if ($params !== false) {
                $request->setParams($params);

                // Execute middleware pipeline
                foreach ($route['middleware'] as $middlewareEntry) {
                    $middlewareInstance = $this->resolveMiddleware($middlewareEntry);
                    $result = $middlewareInstance->handle($request);
                    if ($result === false) {
                        return;
                    }
                }

                // Execute handler — pass route params (e.g. {id}) as extra arguments
                $routeArgs = array_values($params);
                if (is_array($route['handler'])) {
                    [$controllerClass, $method] = $route['handler'];
                    $controller = new $controllerClass();
                    $controller->$method($request, ...$routeArgs);
                } elseif (is_callable($route['handler'])) {
                    ($route['handler'])($request, ...$routeArgs);
                }
                return;
            }
        }

        // No route found
        Response::json(['success' => false, 'message' => 'Route not found'], 404);
    }

    /**
     * Resolve middleware class, supporting parameter syntax:
     *   'CashBook\Middleware\RoleMiddleware:admin,manager'
     */
    private function resolveMiddleware(string $middlewareEntry): object
    {
        if (str_contains($middlewareEntry, ':')) {
            [$className, $paramString] = explode(':', $middlewareEntry, 2);
            $params = explode(',', $paramString);
            return new $className($params);
        }
        return new $middlewareEntry();
    }

    private function matchRoute(string $routePath, string $requestUri): array|false
    {
        $routeParts = explode('/', trim($routePath, '/'));
        $uriParts = explode('/', trim($requestUri, '/'));

        if (count($routeParts) !== count($uriParts)) {
            return false;
        }

        $params = [];
        for ($i = 0; $i < count($routeParts); $i++) {
            if (str_starts_with($routeParts[$i], '{') && str_ends_with($routeParts[$i], '}')) {
                $paramName = trim($routeParts[$i], '{}');
                $params[$paramName] = $uriParts[$i];
            } elseif ($routeParts[$i] !== $uriParts[$i]) {
                return false;
            }
        }

        return $params;
    }
}
