<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

/**
 * Tenant management controller.
 * Only admin users of a tenant can manage tenant settings.
 */
class TenantController extends Controller
{
    /**
     * GET /api/tenant
     * Get current tenant details
     */
    public function show(Request $request): void
    {
        $tenantId = $request->getTenantId();
        $stmt = $this->db->prepare("SELECT * FROM tenants WHERE id = :id");
        $stmt->execute(['id' => $tenantId]);
        $tenant = $stmt->fetch();

        if (!$tenant) {
            Response::notFound('Tenant not found');
            return;
        }

        // Include counts
        $bizCount = $this->db->prepare("SELECT COUNT(*) FROM companies WHERE tenant_id = :tid");
        $bizCount->execute(['tid' => $tenantId]);
        $tenant['business_count'] = (int) $bizCount->fetchColumn();

        $userCount = $this->db->prepare("SELECT COUNT(*) FROM users WHERE tenant_id = :tid AND is_active = TRUE");
        $userCount->execute(['tid' => $tenantId]);
        $tenant['user_count'] = (int) $userCount->fetchColumn();

        Response::success($tenant);
    }

    /**
     * PUT /api/tenant
     * Update tenant settings (admin only)
     */
    public function update(Request $request): void
    {
        $tenantId = $request->getTenantId();
        $data = $request->validate([
            'name' => 'required|min:2'
        ]);

        $stmt = $this->db->prepare(
            "UPDATE tenants SET 
                name = :name, 
                domain = :domain,
                settings = :settings,
                updated_at = NOW()
             WHERE id = :id RETURNING *"
        );
        $stmt->execute([
            'name' => $data['name'],
            'domain' => $request->input('domain'),
            'settings' => json_encode($request->input('settings', [])),
            'id' => $tenantId
        ]);

        $tenant = $stmt->fetch();
        if (!$tenant) {
            Response::notFound('Tenant not found');
            return;
        }

        $this->auditLog($request->getCompanyId(), $request->getUserId(), 'update', 'tenant', $tenantId);
        Response::success($tenant, 'Tenant updated');
    }
}
