<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class ReportController extends Controller
{
    /**
     * GET /reports/income-statement
     * Profit & Loss Statement
     */
    public function incomeStatement(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $dateFrom = $request->query('date_from', date('Y-01-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        // Revenue accounts
        $revenue = $this->getAccountBalances($companyId, 'revenue', $dateFrom, $dateTo);
        $totalRevenue = array_sum(array_column($revenue, 'balance'));

        // Cost of Sales
        $costOfSales = $this->getAccountBalances($companyId, 'expense', $dateFrom, $dateTo, 'cost_of_sales');
        $totalCOGS = array_sum(array_column($costOfSales, 'balance'));

        $grossProfit = $totalRevenue - $totalCOGS;

        // Operating Expenses
        $operatingExpenses = $this->getAccountBalances($companyId, 'expense', $dateFrom, $dateTo, 'operating_expense');
        $totalOpEx = array_sum(array_column($operatingExpenses, 'balance'));

        $operatingProfit = $grossProfit - $totalOpEx;

        // Other Income/Expenses
        $otherRevenue = $this->getAccountBalances($companyId, 'revenue', $dateFrom, $dateTo, 'other_revenue');
        $otherExpenses = $this->getAccountBalances($companyId, 'expense', $dateFrom, $dateTo, 'other_expense');
        $totalOtherRevenue = array_sum(array_column($otherRevenue, 'balance'));
        $totalOtherExpense = array_sum(array_column($otherExpenses, 'balance'));

        $netProfitBeforeTax = $operatingProfit + $totalOtherRevenue - $totalOtherExpense;

        Response::success([
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'revenue' => $revenue,
            'total_revenue' => round($totalRevenue, 2),
            'cost_of_sales' => $costOfSales,
            'total_cost_of_sales' => round($totalCOGS, 2),
            'gross_profit' => round($grossProfit, 2),
            'gross_profit_margin' => $totalRevenue > 0 ? round(($grossProfit / $totalRevenue) * 100, 2) : 0,
            'operating_expenses' => $operatingExpenses,
            'total_operating_expenses' => round($totalOpEx, 2),
            'operating_profit' => round($operatingProfit, 2),
            'other_income' => $otherRevenue,
            'total_other_income' => round($totalOtherRevenue, 2),
            'other_expenses' => $otherExpenses,
            'total_other_expenses' => round($totalOtherExpense, 2),
            'net_profit_before_tax' => round($netProfitBeforeTax, 2),
            'net_profit_margin' => $totalRevenue > 0 ? round(($netProfitBeforeTax / $totalRevenue) * 100, 2) : 0
        ]);
    }

    /**
     * GET /reports/balance-sheet
     */
    public function balanceSheet(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $asOf = $request->query('as_of', date('Y-m-d'));

        // Assets
        $currentAssets = $this->getCumulativeBalances($companyId, 'asset', $asOf, 'current_asset');
        $fixedAssets = $this->getCumulativeBalances($companyId, 'asset', $asOf, 'fixed_asset');
        $allAssets = $this->getCumulativeBalances($companyId, 'asset', $asOf);
        $totalAssets = array_sum(array_column($allAssets, 'balance'));

        // Liabilities
        $currentLiabilities = $this->getCumulativeBalances($companyId, 'liability', $asOf, 'current_liability');
        $longTermLiabilities = $this->getCumulativeBalances($companyId, 'liability', $asOf, 'long_term_liability');
        $allLiabilities = $this->getCumulativeBalances($companyId, 'liability', $asOf);
        $totalLiabilities = array_sum(array_column($allLiabilities, 'balance'));

        // Equity
        $equity = $this->getCumulativeBalances($companyId, 'equity', $asOf);
        $totalEquity = array_sum(array_column($equity, 'balance'));

        // Calculate retained earnings (net profit year to date)
        $yearStart = date('Y') . '-01-01';
        $retainedEarnings = $this->calculateNetProfit($companyId, $yearStart, $asOf);
        $totalEquity += $retainedEarnings;

        Response::success([
            'as_of' => $asOf,
            'assets' => [
                'current_assets' => $currentAssets,
                'total_current_assets' => round(array_sum(array_column($currentAssets, 'balance')), 2),
                'fixed_assets' => $fixedAssets,
                'total_fixed_assets' => round(array_sum(array_column($fixedAssets, 'balance')), 2),
                'total_assets' => round($totalAssets, 2)
            ],
            'liabilities' => [
                'current_liabilities' => $currentLiabilities,
                'total_current_liabilities' => round(array_sum(array_column($currentLiabilities, 'balance')), 2),
                'long_term_liabilities' => $longTermLiabilities,
                'total_long_term_liabilities' => round(array_sum(array_column($longTermLiabilities, 'balance')), 2),
                'total_liabilities' => round($totalLiabilities, 2)
            ],
            'equity' => [
                'accounts' => $equity,
                'retained_earnings' => round($retainedEarnings, 2),
                'total_equity' => round($totalEquity, 2)
            ],
            'total_liabilities_and_equity' => round($totalLiabilities + $totalEquity, 2),
            'is_balanced' => abs($totalAssets - ($totalLiabilities + $totalEquity)) < 0.01
        ]);
    }

    /**
     * GET /reports/cash-flow
     */
    public function cashFlowStatement(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $dateFrom = $request->query('date_from', date('Y-01-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        // Operating Activities
        $netIncome = $this->calculateNetProfit($companyId, $dateFrom, $dateTo);

        // Cash from customers
        $cashFromCustomers = $this->getCashFlowAmount($companyId, $dateFrom, $dateTo, ['1000', '1100', '1800'], 'debit', 'payment');

        // Cash paid to suppliers
        $cashToSuppliers = $this->getCashFlowAmount($companyId, $dateFrom, $dateTo, ['1000', '1100'], 'credit', 'payment');

        // Cash from POS
        $cashFromPOS = $this->getCashFlowAmount($companyId, $dateFrom, $dateTo, ['1000', '1100', '1800'], 'debit', 'pos');

        // Cash paid for payroll
        $cashForPayroll = $this->getCashFlowAmount($companyId, $dateFrom, $dateTo, ['2400'], 'debit', 'payroll');

        $operatingCashFlow = $cashFromCustomers + $cashFromPOS - $cashToSuppliers - $cashForPayroll;

        Response::success([
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'operating_activities' => [
                'net_income' => round($netIncome, 2),
                'cash_from_customers' => round($cashFromCustomers, 2),
                'cash_from_pos_sales' => round($cashFromPOS, 2),
                'cash_paid_to_suppliers' => round($cashToSuppliers, 2),
                'cash_paid_for_payroll' => round($cashForPayroll, 2),
                'net_operating_cash_flow' => round($operatingCashFlow, 2)
            ],
            'summary' => [
                'net_cash_flow' => round($operatingCashFlow, 2)
            ]
        ]);
    }

    /**
     * GET /reports/tax-summary
     */
    public function taxSummary(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $dateFrom = $request->query('date_from', date('Y-01-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        // VAT Summary
        $vatStmt = $this->db->prepare(
            "SELECT 
                COALESCE(SUM(vat_amount), 0) as vat_collected,
                COALESCE(SUM(nhil_amount), 0) as nhil_collected,
                COALESCE(SUM(getfund_amount), 0) as getfund_collected
             FROM invoices WHERE company_id = :cid AND invoice_type = 'sales'
               AND invoice_date BETWEEN :from AND :to AND status != 'voided'"
        );
        $vatStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $vatOutput = $vatStmt->fetch();

        // VAT Input (purchases)
        $vatInputStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(vat_amount), 0) as vat_paid
             FROM invoices WHERE company_id = :cid AND invoice_type = 'purchase'
               AND invoice_date BETWEEN :from AND :to AND status != 'voided'"
        );
        $vatInputStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $vatInput = $vatInputStmt->fetch();

        // POS VAT
        $posVatStmt = $this->db->prepare(
            "SELECT 
                COALESCE(SUM(vat_amount), 0) as vat_collected,
                COALESCE(SUM(nhil_amount), 0) as nhil_collected,
                COALESCE(SUM(getfund_amount), 0) as getfund_collected
             FROM pos_sales WHERE company_id = :cid AND DATE(sale_date) BETWEEN :from AND :to AND status = 'completed'"
        );
        $posVatStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $posVat = $posVatStmt->fetch();

        // PAYE Summary
        $payeStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(ps.paye_tax), 0) as total_paye,
                    COALESCE(SUM(ps.ssnit_employee + ps.ssnit_employer), 0) as total_ssnit,
                    COALESCE(SUM(ps.tier2_employee), 0) as total_tier2
             FROM payslips ps
             JOIN payroll_periods pp ON ps.payroll_period_id = pp.id
             WHERE pp.company_id = :cid AND pp.start_date >= :from AND pp.end_date <= :to"
        );
        $payeStmt->execute(['cid' => $companyId, 'from' => $dateFrom, 'to' => $dateTo]);
        $paye = $payeStmt->fetch();

        $totalVatCollected = (float) $vatOutput['vat_collected'] + (float) $posVat['vat_collected'];
        $vatPayable = $totalVatCollected - (float) $vatInput['vat_paid'];

        Response::success([
            'period' => ['from' => $dateFrom, 'to' => $dateTo],
            'vat' => [
                'output_vat' => round($totalVatCollected, 2),
                'input_vat' => round((float) $vatInput['vat_paid'], 2),
                'vat_payable' => round($vatPayable, 2),
                'nhil_collected' => round((float) $vatOutput['nhil_collected'] + (float) $posVat['nhil_collected'], 2),
                'getfund_collected' => round((float) $vatOutput['getfund_collected'] + (float) $posVat['getfund_collected'], 2)
            ],
            'paye' => [
                'total_paye_tax' => round((float) $paye['total_paye'], 2),
                'total_ssnit' => round((float) $paye['total_ssnit'], 2),
                'total_tier2' => round((float) $paye['total_tier2'], 2)
            ]
        ]);
    }

    /**
     * GET /reports/accounts-receivable-aging
     */
    public function arAging(Request $request): void
    {
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT c.name as contact_name, i.invoice_number, i.invoice_date, i.due_date, i.total_amount, i.balance_due,
                    CURRENT_DATE - i.due_date as days_overdue,
                    CASE 
                        WHEN CURRENT_DATE - i.due_date <= 0 THEN 'current'
                        WHEN CURRENT_DATE - i.due_date <= 30 THEN '1-30'
                        WHEN CURRENT_DATE - i.due_date <= 60 THEN '31-60'
                        WHEN CURRENT_DATE - i.due_date <= 90 THEN '61-90'
                        ELSE '90+'
                    END as aging_bucket
             FROM invoices i JOIN contacts c ON i.contact_id = c.id
             WHERE i.company_id = :cid AND i.invoice_type = 'sales' AND i.balance_due > 0 AND i.status != 'voided'
             ORDER BY i.due_date"
        );
        $stmt->execute(['cid' => $companyId]);
        $invoices = $stmt->fetchAll();

        $buckets = ['current' => 0, '1-30' => 0, '31-60' => 0, '61-90' => 0, '90+' => 0];
        foreach ($invoices as $inv) {
            $buckets[$inv['aging_bucket']] += (float) $inv['balance_due'];
        }

        Response::success([
            'invoices' => $invoices,
            'summary' => $buckets,
            'total_outstanding' => array_sum($buckets)
        ]);
    }

    /**
     * GET /reports/dashboard
     */
    public function dashboard(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $today = date('Y-m-d');
        $monthStart = date('Y-m-01');
        $yearStart = date('Y-01-01');

        // Revenue this month
        $revStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(total_amount), 0) as amount FROM invoices 
             WHERE company_id = :cid AND invoice_type = 'sales' AND status != 'voided' AND invoice_date >= :from"
        );
        $revStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $monthlyRevenue = $revStmt->fetch()['amount'];

        // POS revenue this month
        $posRevStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(total_amount), 0) as amount FROM pos_sales 
             WHERE company_id = :cid AND status = 'completed' AND DATE(sale_date) >= :from"
        );
        $posRevStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $monthlyPOS = $posRevStmt->fetch()['amount'];

        // Expenses this month
        $expStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(total_amount), 0) as amount FROM invoices 
             WHERE company_id = :cid AND invoice_type = 'purchase' AND status != 'voided' AND invoice_date >= :from"
        );
        $expStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $monthlyExpenses = $expStmt->fetch()['amount'];

        // Outstanding receivables
        $arStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(balance_due), 0) as amount FROM invoices 
             WHERE company_id = :cid AND invoice_type = 'sales' AND balance_due > 0 AND status != 'voided'"
        );
        $arStmt->execute(['cid' => $companyId]);
        $totalAR = $arStmt->fetch()['amount'];

        // Outstanding payables
        $apStmt = $this->db->prepare(
            "SELECT COALESCE(SUM(balance_due), 0) as amount FROM invoices 
             WHERE company_id = :cid AND invoice_type = 'purchase' AND balance_due > 0 AND status != 'voided'"
        );
        $apStmt->execute(['cid' => $companyId]);
        $totalAP = $apStmt->fetch()['amount'];

        // Overdue invoices count
        $overdueStmt = $this->db->prepare(
            "SELECT COUNT(*) as count FROM invoices 
             WHERE company_id = :cid AND invoice_type = 'sales' AND balance_due > 0 AND due_date < :today AND status != 'voided'"
        );
        $overdueStmt->execute(['cid' => $companyId, 'today' => $today]);
        $overdueCount = $overdueStmt->fetch()['count'];

        // Today's POS
        $todayPosStmt = $this->db->prepare(
            "SELECT COUNT(*) as count, COALESCE(SUM(total_amount), 0) as total
             FROM pos_sales WHERE company_id = :cid AND DATE(sale_date) = :today AND status = 'completed'"
        );
        $todayPosStmt->execute(['cid' => $companyId, 'today' => $today]);
        $todayPOS = $todayPosStmt->fetch();

        // Low stock alerts
        $lowStockStmt = $this->db->prepare(
            "SELECT COUNT(*) as count FROM products 
             WHERE company_id = :cid AND track_inventory = TRUE AND is_active = TRUE AND quantity_on_hand <= reorder_level"
        );
        $lowStockStmt->execute(['cid' => $companyId]);
        $lowStock = $lowStockStmt->fetch()['count'];

        // Monthly trend (last 6 months)
        $trendStmt = $this->db->prepare(
            "SELECT TO_CHAR(invoice_date, 'YYYY-MM') as month,
                    SUM(CASE WHEN invoice_type = 'sales' THEN total_amount ELSE 0 END) as revenue,
                    SUM(CASE WHEN invoice_type = 'purchase' THEN total_amount ELSE 0 END) as expenses
             FROM invoices WHERE company_id = :cid AND status != 'voided' AND invoice_date >= :from
             GROUP BY TO_CHAR(invoice_date, 'YYYY-MM') ORDER BY month"
        );
        $trendStmt->execute(['cid' => $companyId, 'from' => date('Y-m-d', strtotime('-6 months'))]);
        $trend = $trendStmt->fetchAll();

        // POS Sales Summary (this month breakdown)
        $posSummaryStmt = $this->db->prepare(
            "SELECT COUNT(*) as total_sales, 
                    COALESCE(SUM(total_amount), 0) as total_revenue,
                    COALESCE(AVG(total_amount), 0) as avg_sale,
                    COALESCE(MAX(total_amount), 0) as largest_sale
             FROM pos_sales WHERE company_id = :cid AND status = 'completed' AND DATE(sale_date) >= :from"
        );
        $posSummaryStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $posSummary = $posSummaryStmt->fetch();

        // POS daily trend (last 7 days)
        $posTrendStmt = $this->db->prepare(
            "SELECT DATE(sale_date) as sale_day, COUNT(*) as count, COALESCE(SUM(total_amount), 0) as total
             FROM pos_sales WHERE company_id = :cid AND status = 'completed' AND DATE(sale_date) >= :from
             GROUP BY DATE(sale_date) ORDER BY sale_day"
        );
        $posTrendStmt->execute(['cid' => $companyId, 'from' => date('Y-m-d', strtotime('-7 days'))]);
        $posTrend = $posTrendStmt->fetchAll();

        // Cash Distribution - POS payment methods this month
        $posPaymentStmt = $this->db->prepare(
            "SELECT 
                CASE 
                    WHEN payment_method = 'mobile_money' THEN 'Mobile Money'
                    WHEN payment_method = 'cash' THEN 'Cash'
                    WHEN payment_method = 'card' THEN 'Card/Bank'
                    WHEN payment_method = 'split' THEN 'Split Payment'
                    ELSE INITCAP(REPLACE(payment_method, '_', ' '))
                END as method,
                COUNT(*) as count,
                COALESCE(SUM(total_amount), 0) as total
             FROM pos_sales WHERE company_id = :cid AND status = 'completed' AND DATE(sale_date) >= :from
             GROUP BY payment_method ORDER BY total DESC"
        );
        $posPaymentStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $posPayments = $posPaymentStmt->fetchAll();

        // Cash Distribution - Invoice/Payment methods this month
        $paymentMethodStmt = $this->db->prepare(
            "SELECT 
                CASE 
                    WHEN payment_method = 'mobile_money' THEN 'Mobile Money'
                    WHEN payment_method = 'cash' THEN 'Cash'
                    WHEN payment_method = 'bank_transfer' THEN 'Bank Transfer'
                    WHEN payment_method = 'cheque' THEN 'Cheque'
                    WHEN payment_method = 'card' THEN 'Card/Bank'
                    ELSE INITCAP(REPLACE(COALESCE(payment_method, 'other'), '_', ' '))
                END as method,
                COUNT(*) as count,
                COALESCE(SUM(amount), 0) as total
             FROM payments WHERE company_id = :cid AND status = 'completed' AND payment_date >= :from
             GROUP BY payment_method ORDER BY total DESC"
        );
        $paymentMethodStmt->execute(['cid' => $companyId, 'from' => $monthStart]);
        $invoicePayments = $paymentMethodStmt->fetchAll();

        // Combine POS + Invoice payments into cash distribution
        $cashDistribution = [];
        foreach ($posPayments as $p) {
            $method = $p['method'];
            if (!isset($cashDistribution[$method])) {
                $cashDistribution[$method] = ['method' => $method, 'count' => 0, 'total' => 0];
            }
            $cashDistribution[$method]['count'] += (int) $p['count'];
            $cashDistribution[$method]['total'] += round((float) $p['total'], 2);
        }
        foreach ($invoicePayments as $p) {
            $method = $p['method'];
            if (!isset($cashDistribution[$method])) {
                $cashDistribution[$method] = ['method' => $method, 'count' => 0, 'total' => 0];
            }
            $cashDistribution[$method]['count'] += (int) $p['count'];
            $cashDistribution[$method]['total'] += round((float) $p['total'], 2);
        }
        // Sort by total descending
        usort($cashDistribution, fn($a, $b) => $b['total'] <=> $a['total']);

        Response::success([
            'monthly_revenue' => round((float) $monthlyRevenue + (float) $monthlyPOS, 2),
            'monthly_expenses' => round((float) $monthlyExpenses, 2),
            'monthly_profit' => round(((float) $monthlyRevenue + (float) $monthlyPOS) - (float) $monthlyExpenses, 2),
            'total_receivables' => round((float) $totalAR, 2),
            'total_payables' => round((float) $totalAP, 2),
            'overdue_invoices' => (int) $overdueCount,
            'today_pos_sales' => (int) $todayPOS['count'],
            'today_pos_revenue' => round((float) $todayPOS['total'], 2),
            'low_stock_alerts' => (int) $lowStock,
            'monthly_trend' => $trend,
            'pos_summary' => [
                'month_sales' => (int) $posSummary['total_sales'],
                'month_revenue' => round((float) $posSummary['total_revenue'], 2),
                'avg_sale' => round((float) $posSummary['avg_sale'], 2),
                'largest_sale' => round((float) $posSummary['largest_sale'], 2),
            ],
            'pos_daily_trend' => $posTrend,
            'cash_distribution' => array_values($cashDistribution),
        ]);
    }

    private function getAccountBalances(string $companyId, string $accountType, string $dateFrom, string $dateTo, ?string $subType = null): array
    {
        $subTypeFilter = $subType ? "AND coa.sub_type = :sub_type" : "";
        $stmt = $this->db->prepare(
            "SELECT coa.account_code, coa.account_name, coa.sub_type,
                    COALESCE(SUM(CASE WHEN coa.normal_balance = 'credit' THEN jel.credit_amount - jel.debit_amount ELSE jel.debit_amount - jel.credit_amount END), 0) as balance
             FROM chart_of_accounts coa
             LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
             LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.entry_date BETWEEN :from AND :to
             WHERE coa.company_id = :cid AND coa.account_type = :type AND coa.is_active = TRUE $subTypeFilter
             GROUP BY coa.id, coa.account_code, coa.account_name, coa.sub_type
             HAVING COALESCE(SUM(jel.debit_amount), 0) != 0 OR COALESCE(SUM(jel.credit_amount), 0) != 0
             ORDER BY coa.account_code"
        );
        $params = ['cid' => $companyId, 'type' => $accountType, 'from' => $dateFrom, 'to' => $dateTo];
        if ($subType) $params['sub_type'] = $subType;
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    private function getCumulativeBalances(string $companyId, string $accountType, string $asOf, ?string $subType = null): array
    {
        $subTypeFilter = $subType ? "AND coa.sub_type = :sub_type" : "";
        $stmt = $this->db->prepare(
            "SELECT coa.account_code, coa.account_name, coa.sub_type,
                    coa.opening_balance + COALESCE(SUM(CASE WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount ELSE jel.credit_amount - jel.debit_amount END), 0) as balance
             FROM chart_of_accounts coa
             LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
             LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.entry_date <= :as_of
             WHERE coa.company_id = :cid AND coa.account_type = :type AND coa.is_active = TRUE $subTypeFilter
             GROUP BY coa.id, coa.account_code, coa.account_name, coa.sub_type, coa.opening_balance
             HAVING coa.opening_balance + COALESCE(SUM(CASE WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount ELSE jel.credit_amount - jel.debit_amount END), 0) != 0
             ORDER BY coa.account_code"
        );
        $params = ['cid' => $companyId, 'type' => $accountType, 'as_of' => $asOf];
        if ($subType) $params['sub_type'] = $subType;
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    private function calculateNetProfit(string $companyId, string $dateFrom, string $dateTo): float
    {
        $revenue = $this->getAccountBalances($companyId, 'revenue', $dateFrom, $dateTo);
        $expenses = $this->getAccountBalances($companyId, 'expense', $dateFrom, $dateTo);
        return array_sum(array_column($revenue, 'balance')) - array_sum(array_column($expenses, 'balance'));
    }

    private function getCashFlowAmount(string $companyId, string $dateFrom, string $dateTo, array $accountCodes, string $side, string $source): float
    {
        $placeholders = implode(',', array_map(fn($i) => ":code$i", range(0, count($accountCodes) - 1)));
        $column = $side === 'debit' ? 'debit_amount' : 'credit_amount';

        $stmt = $this->db->prepare(
            "SELECT COALESCE(SUM(jel.{$column}), 0) as amount
             FROM journal_entry_lines jel
             JOIN journal_entries je ON jel.journal_entry_id = je.id
             JOIN chart_of_accounts coa ON jel.account_id = coa.id
             WHERE je.company_id = :cid AND je.status = 'posted' AND je.source = :source
               AND je.entry_date BETWEEN :from AND :to AND coa.account_code IN ($placeholders)"
        );
        $params = ['cid' => $companyId, 'source' => $source, 'from' => $dateFrom, 'to' => $dateTo];
        foreach ($accountCodes as $i => $code) $params["code$i"] = $code;
        $stmt->execute($params);
        return (float) $stmt->fetchColumn();
    }
}
