<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class PayrollController extends Controller
{
    /**
     * GET /payroll/employees
     */
    public function getEmployees(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE e.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('status')) {
            $where .= " AND e.status = :status";
            $params['status'] = $request->query('status');
        }
        if ($request->query('department_id')) {
            $where .= " AND e.department_id = :dept_id";
            $params['dept_id'] = $request->query('department_id');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM employees e $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT e.*, d.name as department_name
                FROM employees e LEFT JOIN departments d ON e.department_id = d.id
                $where ORDER BY e.last_name, e.first_name
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * POST /payroll/employees
     */
    public function createEmployee(Request $request): void
    {
        $data = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'hire_date' => 'required',
            'basic_salary' => 'required|numeric',
            'employment_type' => 'required|in:full_time,part_time,contract,casual'
        ]);

        $companyId = $request->getCompanyId();
        $empNumber = $this->generateNumber('EMP', 'employees', 'employee_number', $companyId);

        $stmt = $this->db->prepare(
            "INSERT INTO employees (company_id, employee_number, first_name, last_name, other_names,
                date_of_birth, gender, marital_status, ghana_card_number, ssnit_number, tin_number,
                email, phone, address, city, region, digital_address, hire_date, employment_type,
                job_title, department_id, basic_salary, payment_method, bank_name, bank_branch,
                bank_account_number, bank_sort_code, mobile_money_provider, mobile_money_number,
                housing_allowance, transport_allowance, meal_allowance, utility_allowance, other_allowances,
                ssnit_employee_rate, ssnit_employer_rate, tier2_employee_rate, tier3_rate)
             VALUES (:company_id, :employee_number, :first_name, :last_name, :other_names,
                :dob, :gender, :marital_status, :ghana_card, :ssnit, :tin,
                :email, :phone, :address, :city, :region, :digital_address, :hire_date, :employment_type,
                :job_title, :department_id, :basic_salary, :payment_method, :bank_name, :bank_branch,
                :bank_account, :sort_code, :momo_provider, :momo_number,
                :housing, :transport, :meal, :utility, :other_allow,
                :ssnit_emp_rate, :ssnit_er_rate, :tier2_rate, :tier3_rate)
             RETURNING *"
        );
        $stmt->execute([
            'company_id' => $companyId,
            'employee_number' => $empNumber,
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'other_names' => $request->input('other_names'),
            'dob' => $request->input('date_of_birth'),
            'gender' => $request->input('gender'),
            'marital_status' => $request->input('marital_status'),
            'ghana_card' => $request->input('ghana_card_number'),
            'ssnit' => $request->input('ssnit_number'),
            'tin' => $request->input('tin_number'),
            'email' => $request->input('email'),
            'phone' => $request->input('phone'),
            'address' => $request->input('address'),
            'city' => $request->input('city'),
            'region' => $request->input('region'),
            'digital_address' => $request->input('digital_address'),
            'hire_date' => $data['hire_date'],
            'employment_type' => $data['employment_type'],
            'job_title' => $request->input('job_title'),
            'department_id' => $request->input('department_id'),
            'basic_salary' => $data['basic_salary'],
            'payment_method' => $request->input('payment_method', 'bank_transfer'),
            'bank_name' => $request->input('bank_name'),
            'bank_branch' => $request->input('bank_branch'),
            'bank_account' => $request->input('bank_account_number'),
            'sort_code' => $request->input('bank_sort_code'),
            'momo_provider' => $request->input('mobile_money_provider'),
            'momo_number' => $request->input('mobile_money_number'),
            'housing' => $request->input('housing_allowance', 0),
            'transport' => $request->input('transport_allowance', 0),
            'meal' => $request->input('meal_allowance', 0),
            'utility' => $request->input('utility_allowance', 0),
            'other_allow' => $request->input('other_allowances', 0),
            'ssnit_emp_rate' => $request->input('ssnit_employee_rate', 5.5),
            'ssnit_er_rate' => $request->input('ssnit_employer_rate', 13),
            'tier2_rate' => $request->input('tier2_employee_rate', 5),
            'tier3_rate' => $request->input('tier3_rate', 0)
        ]);

        $employee = $stmt->fetch();
        $this->auditLog($companyId, $request->getUserId(), 'create', 'employee', $employee['id'], null, $employee);
        Response::created($employee);
    }

    /**
     * GET /payroll/employees/{id}
     */
    public function getEmployee(Request $request): void
    {
        $empId = $request->param('id');
        $stmt = $this->db->prepare(
            "SELECT e.*, d.name as department_name
             FROM employees e LEFT JOIN departments d ON e.department_id = d.id
             WHERE e.id = :id AND e.company_id = :cid"
        );
        $stmt->execute(['id' => $empId, 'cid' => $request->getCompanyId()]);
        $employee = $stmt->fetch();

        if (!$employee) {
            Response::notFound('Employee not found');
            return;
        }

        // Get recent payslips
        $payStmt = $this->db->prepare(
            "SELECT ps.*, pp.period_name FROM payslips ps
             JOIN payroll_periods pp ON ps.payroll_period_id = pp.id
             WHERE ps.employee_id = :eid ORDER BY pp.start_date DESC LIMIT 12"
        );
        $payStmt->execute(['eid' => $empId]);
        $employee['recent_payslips'] = $payStmt->fetchAll();

        Response::success($employee);
    }

    /**
     * PUT /employees/{id}
     */
    public function updateEmployee(Request $request): void
    {
        $empId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM employees WHERE id = :id AND company_id = :cid");
        $stmt->execute(['id' => $empId, 'cid' => $companyId]);
        $employee = $stmt->fetch();

        if (!$employee) {
            Response::notFound('Employee not found');
            return;
        }

        $fields = [
            'first_name', 'last_name', 'other_names', 'date_of_birth', 'gender',
            'marital_status', 'ghana_card_number', 'ssnit_number', 'tin_number',
            'email', 'phone', 'address', 'city', 'region', 'digital_address',
            'hire_date', 'employment_type', 'job_title', 'department_id',
            'basic_salary', 'payment_method', 'bank_name', 'bank_branch',
            'bank_account_number', 'bank_sort_code', 'mobile_money_provider',
            'mobile_money_number', 'housing_allowance', 'transport_allowance',
            'meal_allowance', 'utility_allowance', 'other_allowances',
            'ssnit_employee_rate', 'ssnit_employer_rate', 'tier2_employee_rate',
            'tier3_rate', 'provident_fund_rate', 'loan_deduction', 'other_deductions',
            'status'
        ];

        $setClauses = [];
        $params = ['id' => $empId];
        $body = $request->getBody();

        foreach ($fields as $field) {
            if (array_key_exists($field, $body)) {
                $setClauses[] = "$field = :$field";
                $params[$field] = $body[$field];
            }
        }

        if (empty($setClauses)) {
            Response::error('No fields to update', 400);
            return;
        }

        $setClauses[] = "updated_at = NOW()";
        $sql = "UPDATE employees SET " . implode(', ', $setClauses) . " WHERE id = :id RETURNING *";
        $updateStmt = $this->db->prepare($sql);
        $updateStmt->execute($params);
        $updated = $updateStmt->fetch();

        $this->auditLog($companyId, $request->getUserId(), 'update', 'employee', $empId, $employee, $updated);
        Response::success($updated, 'Employee updated');
    }

    /**
     * POST /payroll/run
     * Run payroll for a period
     */
    public function runPayroll(Request $request): void
    {
        $data = $request->validate([
            'period_name' => 'required',
            'start_date' => 'required',
            'end_date' => 'required',
            'payment_date' => 'required'
        ]);

        $companyId = $request->getCompanyId();

        try {
            $this->db->beginTransaction();

            // Create payroll period
            $periodStmt = $this->db->prepare(
                "INSERT INTO payroll_periods (company_id, period_name, start_date, end_date, payment_date, status, created_by)
                 VALUES (:cid, :name, :start, :end, :pay_date, 'processing', :uid) RETURNING *"
            );
            $periodStmt->execute([
                'cid' => $companyId,
                'name' => $data['period_name'],
                'start' => $data['start_date'],
                'end' => $data['end_date'],
                'pay_date' => $data['payment_date'],
                'uid' => $request->getUserId()
            ]);
            $period = $periodStmt->fetch();

            // Get all active employees
            $empStmt = $this->db->prepare("SELECT * FROM employees WHERE company_id = :cid AND status = 'active'");
            $empStmt->execute(['cid' => $companyId]);
            $employees = $empStmt->fetchAll();

            $totalGross = 0;
            $totalDeductions = 0;
            $totalNet = 0;
            $totalEmployerCost = 0;

            foreach ($employees as $emp) {
                $payslip = $this->calculatePayslip($emp, $period['id']);

                // Insert payslip
                $psStmt = $this->db->prepare(
                    "INSERT INTO payslips (payroll_period_id, employee_id, basic_salary, housing_allowance,
                        transport_allowance, meal_allowance, utility_allowance, overtime_pay, bonus, commission,
                        other_earnings, gross_salary, ssnit_employee, tier2_employee, tier3_voluntary,
                        total_relief, taxable_income, paye_tax, loan_deduction, provident_fund,
                        other_deductions, total_deductions, net_salary, ssnit_employer, total_employer_cost,
                        payment_method)
                     VALUES (:period_id, :emp_id, :basic, :housing, :transport, :meal, :utility,
                        :overtime, :bonus, :commission, :other_earn, :gross, :ssnit_emp, :tier2, :tier3,
                        :relief, :taxable, :paye, :loan, :provident, :other_ded, :total_ded, :net,
                        :ssnit_er, :er_cost, :pay_method)"
                );
                $psStmt->execute([
                    'period_id' => $period['id'],
                    'emp_id' => $emp['id'],
                    'basic' => $payslip['basic_salary'],
                    'housing' => $payslip['housing_allowance'],
                    'transport' => $payslip['transport_allowance'],
                    'meal' => $payslip['meal_allowance'],
                    'utility' => $payslip['utility_allowance'],
                    'overtime' => $payslip['overtime_pay'],
                    'bonus' => $payslip['bonus'],
                    'commission' => $payslip['commission'],
                    'other_earn' => $payslip['other_earnings'],
                    'gross' => $payslip['gross_salary'],
                    'ssnit_emp' => $payslip['ssnit_employee'],
                    'tier2' => $payslip['tier2_employee'],
                    'tier3' => $payslip['tier3_voluntary'],
                    'relief' => $payslip['total_relief'],
                    'taxable' => $payslip['taxable_income'],
                    'paye' => $payslip['paye_tax'],
                    'loan' => $payslip['loan_deduction'],
                    'provident' => $payslip['provident_fund'],
                    'other_ded' => $payslip['other_deductions'],
                    'total_ded' => $payslip['total_deductions'],
                    'net' => $payslip['net_salary'],
                    'ssnit_er' => $payslip['ssnit_employer'],
                    'er_cost' => $payslip['total_employer_cost'],
                    'pay_method' => $emp['payment_method']
                ]);

                $totalGross += $payslip['gross_salary'];
                $totalDeductions += $payslip['total_deductions'];
                $totalNet += $payslip['net_salary'];
                $totalEmployerCost += $payslip['total_employer_cost'];
            }

            // Update period totals
            $this->db->prepare(
                "UPDATE payroll_periods SET total_gross_pay = :gross, total_deductions = :ded,
                    total_net_pay = :net, total_employer_cost = :er_cost, status = 'draft'
                 WHERE id = :id"
            )->execute([
                'gross' => $totalGross, 'ded' => $totalDeductions,
                'net' => $totalNet, 'er_cost' => $totalEmployerCost,
                'id' => $period['id']
            ]);

            $this->db->commit();

            Response::success([
                'period' => $period,
                'summary' => [
                    'employee_count' => count($employees),
                    'total_gross_pay' => round($totalGross, 2),
                    'total_deductions' => round($totalDeductions, 2),
                    'total_net_pay' => round($totalNet, 2),
                    'total_employer_cost' => round($totalEmployerCost, 2)
                ]
            ], 'Payroll processed');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Payroll processing failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /payroll/periods/{id}/approve
     */
    public function approvePayroll(Request $request): void
    {
        $periodId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM payroll_periods WHERE id = :id AND company_id = :cid AND status = 'draft'");
        $stmt->execute(['id' => $periodId, 'cid' => $companyId]);
        $period = $stmt->fetch();

        if (!$period) {
            Response::error('Payroll period not found or cannot be approved', 404);
            return;
        }

        try {
            $this->db->beginTransaction();

            // Create journal entry for payroll
            $this->createPayrollJournalEntry($companyId, $period, $request->getUserId());

            $this->db->prepare(
                "UPDATE payroll_periods SET status = 'approved', approved_by = :uid, approved_at = NOW() WHERE id = :id"
            )->execute(['uid' => $request->getUserId(), 'id' => $periodId]);

            $this->db->commit();
            $this->auditLog($companyId, $request->getUserId(), 'approve', 'payroll', $periodId);
            Response::success(null, 'Payroll approved');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to approve payroll: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /payroll/periods
     */
    public function getPayrollPeriods(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare(
            "SELECT pp.*, u.first_name || ' ' || u.last_name as created_by_name,
                    (SELECT COUNT(*) FROM payslips WHERE payroll_period_id = pp.id) as employee_count
             FROM payroll_periods pp LEFT JOIN users u ON pp.created_by = u.id
             WHERE pp.company_id = :cid ORDER BY pp.start_date DESC"
        );
        $stmt->execute(['cid' => $companyId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * GET /payroll/periods/{id}/payslips
     */
    public function getPayslips(Request $request): void
    {
        $periodId = $request->param('id');
        $stmt = $this->db->prepare(
            "SELECT ps.*, e.employee_number, e.first_name, e.last_name, e.job_title, d.name as department_name
             FROM payslips ps
             JOIN employees e ON ps.employee_id = e.id
             LEFT JOIN departments d ON e.department_id = d.id
             WHERE ps.payroll_period_id = :pid ORDER BY e.last_name, e.first_name"
        );
        $stmt->execute(['pid' => $periodId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * GET /payroll/tax-brackets
     */
    public function getTaxBrackets(Request $request): void
    {
        $stmt = $this->db->prepare("SELECT * FROM paye_tax_brackets WHERE is_active = TRUE ORDER BY bracket_order");
        $stmt->execute();
        Response::success($stmt->fetchAll());
    }

    /**
     * Calculate individual payslip with Ghana PAYE
     */
    private function calculatePayslip(array $emp, string $periodId): array
    {
        $basic = (float) $emp['basic_salary'];
        $housing = (float) $emp['housing_allowance'];
        $transport = (float) $emp['transport_allowance'];
        $meal = (float) $emp['meal_allowance'];
        $utility = (float) $emp['utility_allowance'];
        $otherAllow = (float) $emp['other_allowances'];

        $grossSalary = $basic + $housing + $transport + $meal + $utility + $otherAllow;

        // SSNIT Employee Contribution (5.5% of basic)
        $ssnitEmployee = round($basic * ((float) $emp['ssnit_employee_rate'] / 100), 2);

        // Tier 2 Employee Contribution (5% of basic)
        $tier2Employee = round($basic * ((float) $emp['tier2_employee_rate'] / 100), 2);

        // Tier 3 Voluntary
        $tier3 = round($basic * ((float) $emp['tier3_rate'] / 100), 2);

        // SSNIT Employer Contribution (13% of basic)
        $ssnitEmployer = round($basic * ((float) $emp['ssnit_employer_rate'] / 100), 2);

        // Tax Relief = SSNIT Employee (5.5%) + Consolidated Relief (already factored in brackets)
        $totalRelief = $ssnitEmployee;

        // Taxable Income = Gross - SSNIT Employee
        $taxableIncome = $grossSalary - $ssnitEmployee;

        // Calculate PAYE using Ghana tax brackets (monthly)
        $payeTax = $this->calculatePAYE($taxableIncome);

        // Other deductions
        $loanDeduction = (float) $emp['loan_deduction'];
        $providentFund = (float) ($emp['provident_fund_rate'] ?? 0) / 100 * $basic;
        $otherDeductions = (float) $emp['other_deductions'];

        $totalDeductions = $ssnitEmployee + $tier2Employee + $tier3 + $payeTax + $loanDeduction + $providentFund + $otherDeductions;
        $netSalary = $grossSalary - $totalDeductions;

        $totalEmployerCost = $grossSalary + $ssnitEmployer;

        return [
            'basic_salary' => $basic,
            'housing_allowance' => $housing,
            'transport_allowance' => $transport,
            'meal_allowance' => $meal,
            'utility_allowance' => $utility,
            'overtime_pay' => 0,
            'bonus' => 0,
            'commission' => 0,
            'other_earnings' => $otherAllow,
            'gross_salary' => round($grossSalary, 2),
            'ssnit_employee' => $ssnitEmployee,
            'tier2_employee' => $tier2Employee,
            'tier3_voluntary' => $tier3,
            'total_relief' => round($totalRelief, 2),
            'taxable_income' => round($taxableIncome, 2),
            'paye_tax' => round($payeTax, 2),
            'loan_deduction' => round($loanDeduction, 2),
            'provident_fund' => round($providentFund, 2),
            'other_deductions' => round($otherDeductions, 2),
            'total_deductions' => round($totalDeductions, 2),
            'net_salary' => round($netSalary, 2),
            'ssnit_employer' => $ssnitEmployer,
            'total_employer_cost' => round($totalEmployerCost, 2)
        ];
    }

    /**
     * Calculate Ghana PAYE Tax (Monthly)
     * Based on GRA tax brackets
     */
    private function calculatePAYE(float $monthlyTaxableIncome): float
    {
        // Ghana PAYE Monthly Tax Brackets (2024/2025)
        $brackets = [
            ['limit' => 490,       'rate' => 0],      // First GHS 490
            ['limit' => 110,       'rate' => 5],      // Next GHS 110
            ['limit' => 130,       'rate' => 10],     // Next GHS 130
            ['limit' => 3166.67,   'rate' => 17.5],   // Next GHS 3,166.67
            ['limit' => 16103.33,  'rate' => 25],     // Next GHS 16,103.33
            ['limit' => 30000,     'rate' => 30],     // Next GHS 30,000
            ['limit' => PHP_FLOAT_MAX, 'rate' => 35], // Exceeding
        ];

        $tax = 0;
        $remaining = $monthlyTaxableIncome;

        foreach ($brackets as $bracket) {
            if ($remaining <= 0) break;

            $taxableInBracket = min($remaining, $bracket['limit']);
            $tax += $taxableInBracket * ($bracket['rate'] / 100);
            $remaining -= $taxableInBracket;
        }

        return round($tax, 2);
    }

    private function createPayrollJournalEntry(string $companyId, array $period, string $userId): void
    {
        // Get payslip totals
        $stmt = $this->db->prepare(
            "SELECT SUM(gross_salary) as total_gross, SUM(ssnit_employee) as total_ssnit_emp,
                    SUM(tier2_employee) as total_tier2, SUM(paye_tax) as total_paye,
                    SUM(net_salary) as total_net, SUM(ssnit_employer) as total_ssnit_er
             FROM payslips WHERE payroll_period_id = :pid"
        );
        $stmt->execute(['pid' => $period['id']]);
        $totals = $stmt->fetch();

        $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

        // Journal Entry: Debit Salary Expense, Credit various payables
        $jeStmt = $this->db->prepare(
            "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, source, source_id, total_debit, total_credit, status, created_by, posted_at)
             VALUES (:cid, :num, :date, :desc, 'payroll', :sid, :debit, :credit, 'posted', :uid, NOW())
             RETURNING id"
        );

        $totalDebit = (float) $totals['total_gross'] + (float) $totals['total_ssnit_er'];
        $jeStmt->execute([
            'cid' => $companyId, 'num' => $entryNumber,
            'date' => $period['payment_date'],
            'desc' => "Payroll - {$period['period_name']}",
            'sid' => $period['id'],
            'debit' => $totalDebit, 'credit' => $totalDebit,
            'uid' => $userId
        ]);
        $je = $jeStmt->fetch();

        $lineStmt = $this->db->prepare(
            "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description)
             VALUES (:je_id, :account_id, :debit, :credit, :desc)"
        );

        $getAccount = function (string $code) use ($companyId) {
            $s = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
            $s->execute(['cid' => $companyId, 'code' => $code]);
            return $s->fetchColumn();
        };

        // Debit: Salary Expense (5100)
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('5100'), 'debit' => $totals['total_gross'], 'credit' => 0, 'desc' => 'Salaries & Wages']);

        // Debit: Employer SSNIT (5110)
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('5110'), 'debit' => $totals['total_ssnit_er'], 'credit' => 0, 'desc' => 'Employer SSNIT Contribution']);

        // Credit: Salaries Payable (2400)
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('2400'), 'debit' => 0, 'credit' => $totals['total_net'], 'desc' => 'Net Salaries Payable']);

        // Credit: PAYE Tax Payable (2200)
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('2200'), 'debit' => 0, 'credit' => $totals['total_paye'], 'desc' => 'PAYE Tax']);

        // Credit: SSNIT Payable (2210) - includes both employee + employer
        $totalSSNIT = (float) $totals['total_ssnit_emp'] + (float) $totals['total_ssnit_er'];
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('2210'), 'debit' => 0, 'credit' => $totalSSNIT, 'desc' => 'SSNIT Contributions']);

        // Credit: Tier 2 Payable (2220)
        $lineStmt->execute(['je_id' => $je['id'], 'account_id' => $getAccount('2220'), 'debit' => 0, 'credit' => $totals['total_tier2'], 'desc' => 'Tier 2 Pension']);

        // Update period with journal entry
        $this->db->prepare("UPDATE payroll_periods SET journal_entry_id = :je WHERE id = :id")->execute(['je' => $je['id'], 'id' => $period['id']]);
    }
}
