<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class InventoryController extends Controller
{
    /**
     * GET /products
     */
    public function index(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE p.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('category_id')) {
            $where .= " AND p.category_id = :cat_id";
            $params['cat_id'] = $request->query('category_id');
        }
        if ($request->query('search')) {
            $where .= " AND (p.name ILIKE :search OR p.sku ILIKE :search2 OR p.barcode = :barcode)";
            $params['search'] = '%' . $request->query('search') . '%';
            $params['search2'] = '%' . $request->query('search') . '%';
            $params['barcode'] = $request->query('search');
        }
        if ($request->query('active') !== null) {
            $where .= " AND p.is_active = :active";
            $params['active'] = $request->query('active') === 'true';
        }
        if ($request->query('low_stock') === 'true') {
            $where .= " AND p.track_inventory = TRUE AND p.quantity_on_hand <= p.reorder_level";
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM products p $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT p.*, pc.name as category_name
                FROM products p LEFT JOIN product_categories pc ON p.category_id = pc.id
                $where ORDER BY p.name
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * POST /products
     */
    public function create(Request $request): void
    {
        $data = $request->validate([
            'name' => 'required',
            'selling_price' => 'required|numeric'
        ]);

        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "INSERT INTO products (company_id, category_id, sku, barcode, name, description, unit_of_measure,
                cost_price, selling_price, wholesale_price, tax_inclusive, tax_rate, track_inventory,
                quantity_on_hand, reorder_level, reorder_quantity, is_service, income_account_id,
                expense_account_id, inventory_account_id, image_url)
             VALUES (:company_id, :category_id, :sku, :barcode, :name, :description, :uom,
                :cost_price, :selling_price, :wholesale_price, :tax_inclusive, :tax_rate, :track_inventory,
                :qty_on_hand, :reorder_level, :reorder_qty, :is_service, :income_acct,
                :expense_acct, :inventory_acct, :image_url)
             RETURNING *"
        );

        $sku = $request->input('sku');
        if (!$sku) {
            $sku = strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $data['name']), 0, 6)) . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
        }

        $stmt->execute([
            'company_id' => $companyId,
            'category_id' => $request->input('category_id') ?: null,
            'sku' => $sku,
            'barcode' => $request->input('barcode') ?: null,
            'name' => $data['name'],
            'description' => $request->input('description'),
            'uom' => $request->input('unit_of_measure', 'unit'),
            'cost_price' => $request->input('cost_price', 0),
            'selling_price' => $data['selling_price'],
            'wholesale_price' => $request->input('wholesale_price') ?: null,
            'tax_inclusive' => $request->input('tax_inclusive', false) ? 't' : 'f',
            'tax_rate' => $request->input('tax_rate', 0),
            'track_inventory' => $request->input('track_inventory', true) ? 't' : 'f',
            'qty_on_hand' => $request->input('quantity_on_hand', 0),
            'reorder_level' => $request->input('reorder_level', 0),
            'reorder_qty' => $request->input('reorder_quantity', 0),
            'is_service' => $request->input('is_service', false) ? 't' : 'f',
            'income_acct' => $request->input('income_account_id') ?: null,
            'expense_acct' => $request->input('expense_account_id') ?: null,
            'inventory_acct' => $request->input('inventory_account_id') ?: null,
            'image_url' => $request->input('image_url') ?: null
        ]);

        $product = $stmt->fetch();
        $this->auditLog($companyId, $request->getUserId(), 'create', 'product', $product['id']);
        Response::created($product);
    }

    /**
     * GET /products/{id}
     */
    public function show(Request $request): void
    {
        $stmt = $this->db->prepare(
            "SELECT p.*, pc.name as category_name
             FROM products p LEFT JOIN product_categories pc ON p.category_id = pc.id
             WHERE p.id = :id AND p.company_id = :cid"
        );
        $stmt->execute(['id' => $request->param('id'), 'cid' => $request->getCompanyId()]);
        $product = $stmt->fetch();

        if (!$product) {
            Response::notFound('Product not found');
            return;
        }

        // Recent movements
        $movStmt = $this->db->prepare(
            "SELECT * FROM inventory_movements WHERE product_id = :pid ORDER BY created_at DESC LIMIT 20"
        );
        $movStmt->execute(['pid' => $product['id']]);
        $product['movements'] = $movStmt->fetchAll();

        Response::success($product);
    }

    /**
     * PUT /products/{id}
     */
    public function update(Request $request): void
    {
        $productId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM products WHERE id = :id AND company_id = :cid");
        $stmt->execute(['id' => $productId, 'cid' => $companyId]);
        $existing = $stmt->fetch();
        if (!$existing) {
            Response::notFound('Product not found');
            return;
        }

        // Accept both frontend field names (unit, stock_quantity) and backend names
        $uom = $request->input('unit_of_measure') ?? $request->input('unit') ?? $existing['unit_of_measure'];
        $qtyOnHand = $request->input('quantity_on_hand') ?? $request->input('stock_quantity') ?? $existing['quantity_on_hand'];

        $this->db->prepare(
            "UPDATE products SET name = :name, description = :desc, category_id = :cat, sku = :sku,
                barcode = :barcode, unit_of_measure = :uom, cost_price = :cost, selling_price = :price,
                wholesale_price = :wholesale, tax_rate = :tax_rate, track_inventory = :track,
                quantity_on_hand = :qty_on_hand, reorder_level = :reorder, reorder_quantity = :reorder_qty,
                is_active = :active, is_service = :service, image_url = :image, updated_at = NOW()
             WHERE id = :id AND company_id = :cid"
        )->execute([
            'id' => $productId, 'cid' => $companyId,
            'name' => $request->input('name', $existing['name']),
            'desc' => $request->input('description', $existing['description']),
            'cat' => $request->input('category_id', $existing['category_id']) ?: null,
            'sku' => $request->input('sku', $existing['sku']),
            'barcode' => $request->input('barcode', $existing['barcode']) ?: null,
            'uom' => $uom,
            'cost' => $request->input('cost_price', $existing['cost_price']),
            'price' => $request->input('selling_price', $existing['selling_price']),
            'wholesale' => $request->input('wholesale_price', $existing['wholesale_price']),
            'tax_rate' => $request->input('tax_rate', $existing['tax_rate'] ?? 0),
            'track' => filter_var($request->input('track_inventory', $existing['track_inventory'] ?? true), FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
            'qty_on_hand' => $qtyOnHand,
            'reorder' => $request->input('reorder_level', $existing['reorder_level'] ?? 0),
            'reorder_qty' => $request->input('reorder_quantity', $existing['reorder_quantity'] ?? 0),
            'active' => filter_var($request->input('is_active', $existing['is_active'] ?? true), FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
            'service' => filter_var($request->input('is_service', $existing['is_service'] ?? false), FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
            'image' => $request->input('image_url', $existing['image_url'])
        ]);

        // Return the updated product
        $updated = $this->db->prepare("SELECT p.*, pc.name as category_name FROM products p LEFT JOIN product_categories pc ON p.category_id = pc.id WHERE p.id = :id");
        $updated->execute(['id' => $productId]);
        Response::success($updated->fetch(), 'Product updated');
    }

    /**
     * POST /inventory/adjust
     */
    public function adjustStock(Request $request): void
    {
        $data = $request->validate([
            'product_id' => 'required',
            'quantity' => 'required|numeric',
            'reason' => 'required'
        ]);

        $companyId = $request->getCompanyId();

        $prodStmt = $this->db->prepare("SELECT * FROM products WHERE id = :id AND company_id = :cid");
        $prodStmt->execute(['id' => $data['product_id'], 'cid' => $companyId]);
        $product = $prodStmt->fetch();

        if (!$product) {
            Response::notFound('Product not found');
            return;
        }

        $newQty = (float) $product['quantity_on_hand'] + (float) $data['quantity'];

        $this->db->prepare("UPDATE products SET quantity_on_hand = :qty, updated_at = NOW() WHERE id = :id")
            ->execute(['qty' => $newQty, 'id' => $data['product_id']]);

        $this->db->prepare(
            "INSERT INTO inventory_movements (company_id, product_id, movement_type, quantity, notes, quantity_before, quantity_after, created_by)
             VALUES (:cid, :pid, 'adjustment', :qty, :notes, :before, :after, :uid)"
        )->execute([
            'cid' => $companyId,
            'pid' => $data['product_id'],
            'qty' => $data['quantity'],
            'notes' => $data['reason'],
            'before' => $product['quantity_on_hand'],
            'after' => $newQty,
            'uid' => $request->getUserId()
        ]);

        $this->auditLog($companyId, $request->getUserId(), 'adjust', 'inventory', $data['product_id']);
        Response::success(['quantity_on_hand' => $newQty], 'Stock adjusted');
    }

    /**
     * GET /inventory/low-stock
     */
    public function lowStock(Request $request): void
    {
        $stmt = $this->db->prepare(
            "SELECT p.*, pc.name as category_name
             FROM products p LEFT JOIN product_categories pc ON p.category_id = pc.id
             WHERE p.company_id = :cid AND p.track_inventory = TRUE AND p.is_active = TRUE
               AND p.quantity_on_hand <= p.reorder_level
             ORDER BY (p.quantity_on_hand - p.reorder_level) ASC"
        );
        $stmt->execute(['cid' => $request->getCompanyId()]);
        Response::success($stmt->fetchAll());
    }

    /**
     * GET /product-categories
     */
    public function getCategories(Request $request): void
    {
        $stmt = $this->db->prepare(
            "SELECT * FROM product_categories WHERE company_id = :cid ORDER BY sort_order, name"
        );
        $stmt->execute(['cid' => $request->getCompanyId()]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /product-categories
     */
    public function createCategory(Request $request): void
    {
        $data = $request->validate(['name' => 'required']);

        $stmt = $this->db->prepare(
            "INSERT INTO product_categories (company_id, name, parent_id, description) VALUES (:cid, :name, :parent, :desc) RETURNING *"
        );
        $stmt->execute([
            'cid' => $request->getCompanyId(),
            'name' => $data['name'],
            'parent' => $request->input('parent_id'),
            'desc' => $request->input('description')
        ]);
        Response::created($stmt->fetch());
    }

    /**
     * PUT /product-categories/{id}
     */
    public function updateCategory(Request $request): void
    {
        $catId = $request->param('id');
        $companyId = $request->getCompanyId();

        $data = $request->validate(['name' => 'required']);

        $stmt = $this->db->prepare(
            "UPDATE product_categories SET name = :name, description = :desc, parent_id = :parent, updated_at = NOW()
             WHERE id = :id AND company_id = :cid RETURNING *"
        );
        $stmt->execute([
            'name' => $data['name'],
            'desc' => $request->input('description'),
            'parent' => $request->input('parent_id') ?: null,
            'id' => $catId,
            'cid' => $companyId
        ]);

        $cat = $stmt->fetch();
        if (!$cat) {
            Response::notFound('Category not found');
            return;
        }
        Response::success($cat, 'Category updated');
    }

    /**
     * DELETE /product-categories/{id}
     */
    public function deleteCategory(Request $request): void
    {
        $catId = $request->param('id');
        $companyId = $request->getCompanyId();

        // Check if any products use this category
        $check = $this->db->prepare("SELECT COUNT(*) FROM products WHERE category_id = :id AND company_id = :cid");
        $check->execute(['id' => $catId, 'cid' => $companyId]);
        $count = (int) $check->fetchColumn();
        if ($count > 0) {
            Response::error("Cannot delete category — $count product(s) are assigned to it", 422);
            return;
        }

        $stmt = $this->db->prepare("DELETE FROM product_categories WHERE id = :id AND company_id = :cid");
        $stmt->execute(['id' => $catId, 'cid' => $companyId]);
        Response::success(null, 'Category deleted');
    }

    /**
     * GET /products/barcode/{barcode}
     * Quick lookup for POS barcode scanning
     */
    public function lookupBarcode(Request $request): void
    {
        $barcode = $request->param('barcode');
        $stmt = $this->db->prepare(
            "SELECT id, name, sku, barcode, selling_price, cost_price, quantity_on_hand, tax_rate, unit_of_measure, image_url
             FROM products WHERE (barcode = :barcode OR sku = :sku) AND company_id = :cid AND is_active = TRUE LIMIT 1"
        );
        $stmt->execute(['barcode' => $barcode, 'sku' => $barcode, 'cid' => $request->getCompanyId()]);
        $product = $stmt->fetch();

        if (!$product) {
            Response::notFound('Product not found');
            return;
        }
        Response::success($product);
    }

    /**
     * List inventory movements with optional filters
     */
    public function movements(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = ['im.company_id = :cid'];
        $params = ['cid' => $companyId];

        // Optional filters
        $productId = $request->input('product_id');
        if ($productId) {
            $where[] = 'im.product_id = :pid';
            $params['pid'] = $productId;
        }

        $type = $request->input('movement_type');
        if ($type) {
            $where[] = 'im.movement_type = :type';
            $params['type'] = $type;
        }

        $whereClause = implode(' AND ', $where);

        // Count
        $countStmt = $this->db->prepare("SELECT COUNT(*) as total FROM inventory_movements im WHERE $whereClause");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetch()['total'];

        // Fetch with product name
        $sql = "SELECT im.*, p.name as product_name, p.sku
                FROM inventory_movements im
                LEFT JOIN products p ON im.product_id = p.id
                WHERE $whereClause
                ORDER BY im.created_at DESC
                LIMIT :limit OFFSET :offset";
        $stmt = $this->db->prepare($sql);
        foreach ($params as $k => $v) {
            $stmt->bindValue(":$k", $v);
        }
        $stmt->bindValue(':limit', $pagination['per_page'], \PDO::PARAM_INT);
        $stmt->bindValue(':offset', $pagination['offset'], \PDO::PARAM_INT);
        $stmt->execute();
        $movements = $stmt->fetchAll();

        Response::paginated($movements, $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * POST /products/import
     * Bulk import products from parsed Excel data (JSON array)
     */
    public function import(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $userId = $request->getUserId();
        $products = $request->input('products');

        if (!$products || !is_array($products) || count($products) === 0) {
            Response::error('No products provided', 422);
            return;
        }

        if (count($products) > 500) {
            Response::error('Maximum 500 products per import', 422);
            return;
        }

        // Load existing categories for name-to-id mapping
        $catStmt = $this->db->prepare("SELECT id, LOWER(name) as name FROM product_categories WHERE company_id = :cid");
        $catStmt->execute(['cid' => $companyId]);
        $categoryMap = [];
        foreach ($catStmt->fetchAll() as $cat) {
            $categoryMap[$cat['name']] = $cat['id'];
        }

        $insertStmt = $this->db->prepare(
            "INSERT INTO products (company_id, category_id, sku, barcode, name, description, unit_of_measure,
                cost_price, selling_price, wholesale_price, tax_inclusive, tax_rate, track_inventory,
                quantity_on_hand, reorder_level, reorder_quantity, is_service)
             VALUES (:company_id, :category_id, :sku, :barcode, :name, :description, :uom,
                :cost_price, :selling_price, :wholesale_price, :tax_inclusive, :tax_rate, :track_inventory,
                :qty_on_hand, :reorder_level, :reorder_qty, :is_service)
             ON CONFLICT (company_id, sku) DO UPDATE SET
                name = EXCLUDED.name,
                description = EXCLUDED.description,
                category_id = EXCLUDED.category_id,
                barcode = EXCLUDED.barcode,
                unit_of_measure = EXCLUDED.unit_of_measure,
                cost_price = EXCLUDED.cost_price,
                selling_price = EXCLUDED.selling_price,
                wholesale_price = EXCLUDED.wholesale_price,
                quantity_on_hand = EXCLUDED.quantity_on_hand,
                reorder_level = EXCLUDED.reorder_level,
                reorder_quantity = EXCLUDED.reorder_quantity,
                updated_at = NOW()
             RETURNING id"
        );

        $imported = 0;
        $errors = [];

        $this->db->beginTransaction();
        try {
            foreach ($products as $i => $row) {
                $name = trim($row['name'] ?? '');
                if (!$name) {
                    $errors[] = "Row " . ($i + 1) . ": Product name is required";
                    continue;
                }

                $sellingPrice = floatval($row['selling_price'] ?? 0);
                if ($sellingPrice <= 0) {
                    $errors[] = "Row " . ($i + 1) . " ($name): Selling price must be greater than 0";
                    continue;
                }

                // Auto-generate SKU if not provided
                $sku = trim($row['sku'] ?? '');
                if (!$sku) {
                    $sku = strtoupper(substr(preg_replace('/[^a-zA-Z0-9]/', '', $name), 0, 6)) . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
                }

                // Resolve category by name or ID
                $categoryId = null;
                $catValue = trim($row['category'] ?? $row['category_id'] ?? '');
                if ($catValue) {
                    // Check if it's a UUID
                    if (preg_match('/^[0-9a-f]{8}-/', $catValue)) {
                        $categoryId = $catValue;
                    } else {
                        // Look up by name (case-insensitive)
                        $categoryId = $categoryMap[strtolower($catValue)] ?? null;
                        // Auto-create category if not found
                        if (!$categoryId) {
                            $newCat = $this->db->prepare(
                                "INSERT INTO product_categories (company_id, name) VALUES (:cid, :name) RETURNING id"
                            );
                            $newCat->execute(['cid' => $companyId, 'name' => $catValue]);
                            $categoryId = $newCat->fetch()['id'];
                            $categoryMap[strtolower($catValue)] = $categoryId;
                        }
                    }
                }

                try {
                    $insertStmt->execute([
                        'company_id' => $companyId,
                        'category_id' => $categoryId,
                        'sku' => $sku,
                        'barcode' => trim($row['barcode'] ?? '') ?: null,
                        'name' => $name,
                        'description' => trim($row['description'] ?? '') ?: null,
                        'uom' => trim($row['unit_of_measure'] ?? $row['unit'] ?? 'piece'),
                        'cost_price' => floatval($row['cost_price'] ?? 0),
                        'selling_price' => $sellingPrice,
                        'wholesale_price' => !empty($row['wholesale_price']) ? floatval($row['wholesale_price']) : null,
                        'tax_inclusive' => filter_var($row['tax_inclusive'] ?? false, FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
                        'tax_rate' => floatval($row['tax_rate'] ?? 0),
                        'track_inventory' => filter_var($row['track_inventory'] ?? true, FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
                        'qty_on_hand' => floatval($row['quantity_on_hand'] ?? $row['stock'] ?? 0),
                        'reorder_level' => intval($row['reorder_level'] ?? 10),
                        'reorder_qty' => intval($row['reorder_quantity'] ?? 0),
                        'is_service' => filter_var($row['is_service'] ?? false, FILTER_VALIDATE_BOOLEAN) ? 't' : 'f',
                    ]);
                    $imported++;
                } catch (\Exception $e) {
                    $errors[] = "Row " . ($i + 1) . " ($name): " . $e->getMessage();
                }
            }

            $this->db->commit();
            $this->auditLog($companyId, $userId, 'import', 'products', null);

            Response::success([
                'imported' => $imported,
                'total' => count($products),
                'errors' => $errors,
            ], "$imported product(s) imported successfully" . (count($errors) ? ', ' . count($errors) . ' error(s)' : ''));

        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Import failed: ' . $e->getMessage(), 500);
        }
    }
}
