<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class ContactController extends Controller
{
    public function index(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('type')) {
            $where .= " AND (contact_type = :type OR contact_type = 'both')";
            $params['type'] = $request->query('type');
        }
        if ($request->query('search')) {
            $where .= " AND (name ILIKE :search OR email ILIKE :search2 OR phone LIKE :phone)";
            $params['search'] = '%' . $request->query('search') . '%';
            $params['search2'] = '%' . $request->query('search') . '%';
            $params['phone'] = '%' . $request->query('search') . '%';
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM contacts $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT * FROM contacts $where ORDER BY name LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        Response::paginated($stmt->fetchAll(), $total, $pagination['page'], $pagination['per_page']);
    }

    public function create(Request $request): void
    {
        $data = $request->validate([
            'name' => 'required',
            'contact_type' => 'required|in:customer,supplier,both'
        ]);

        $stmt = $this->db->prepare(
            "INSERT INTO contacts (company_id, contact_type, name, trading_name, tin_number, vat_number,
                email, phone, mobile, address, city, region, digital_address, payment_terms, credit_limit, notes)
             VALUES (:cid, :type, :name, :trading, :tin, :vat, :email, :phone, :mobile, :address, :city,
                :region, :digital, :terms, :credit, :notes)
             RETURNING *"
        );
        $stmt->execute([
            'cid' => $request->getCompanyId(),
            'type' => $data['contact_type'],
            'name' => $data['name'],
            'trading' => $request->input('trading_name'),
            'tin' => $request->input('tin_number'),
            'vat' => $request->input('vat_number'),
            'email' => $request->input('email'),
            'phone' => $request->input('phone'),
            'mobile' => $request->input('mobile'),
            'address' => $request->input('address'),
            'city' => $request->input('city'),
            'region' => $request->input('region'),
            'digital' => $request->input('digital_address'),
            'terms' => $request->input('payment_terms', 30),
            'credit' => $request->input('credit_limit', 0),
            'notes' => $request->input('notes')
        ]);
        Response::created($stmt->fetch());
    }

    public function show(Request $request): void
    {
        $stmt = $this->db->prepare("SELECT * FROM contacts WHERE id = :id AND company_id = :cid");
        $stmt->execute(['id' => $request->param('id'), 'cid' => $request->getCompanyId()]);
        $contact = $stmt->fetch();

        if (!$contact) {
            Response::notFound('Contact not found');
            return;
        }

        // Get recent invoices
        $invStmt = $this->db->prepare(
            "SELECT id, invoice_number, invoice_type, invoice_date, due_date, total_amount, balance_due, status
             FROM invoices WHERE contact_id = :id ORDER BY invoice_date DESC LIMIT 10"
        );
        $invStmt->execute(['id' => $contact['id']]);
        $contact['recent_invoices'] = $invStmt->fetchAll();

        Response::success($contact);
    }

    public function update(Request $request): void
    {
        $contactId = $request->param('id');
        $companyId = $request->getCompanyId();

        $this->db->prepare(
            "UPDATE contacts SET name = :name, trading_name = :trading, contact_type = :type,
                tin_number = :tin, vat_number = :vat, email = :email, phone = :phone, mobile = :mobile,
                address = :address, city = :city, region = :region, digital_address = :digital,
                payment_terms = :terms, credit_limit = :credit, notes = :notes, updated_at = NOW()
             WHERE id = :id AND company_id = :cid"
        )->execute([
            'id' => $contactId, 'cid' => $companyId,
            'name' => $request->input('name'),
            'trading' => $request->input('trading_name'),
            'type' => $request->input('contact_type'),
            'tin' => $request->input('tin_number'),
            'vat' => $request->input('vat_number'),
            'email' => $request->input('email'),
            'phone' => $request->input('phone'),
            'mobile' => $request->input('mobile'),
            'address' => $request->input('address'),
            'city' => $request->input('city'),
            'region' => $request->input('region'),
            'digital' => $request->input('digital_address'),
            'terms' => $request->input('payment_terms'),
            'credit' => $request->input('credit_limit'),
            'notes' => $request->input('notes')
        ]);
        Response::success(null, 'Contact updated');
    }

    public function delete(Request $request): void
    {
        $this->db->prepare("UPDATE contacts SET is_active = FALSE WHERE id = :id AND company_id = :cid")
            ->execute(['id' => $request->param('id'), 'cid' => $request->getCompanyId()]);
        Response::success(null, 'Contact deactivated');
    }
}
