<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

/**
 * Business (company) management within a tenant.
 * Admin can CRUD businesses; users can list those they have access to.
 */
class BusinessController extends Controller
{
    /**
     * GET /api/businesses
     * List all businesses for the current tenant that the user has access to
     */
    public function index(Request $request): void
    {
        $tenantId = $request->getTenantId();
        $userId = $request->getUserId();
        $userRole = $request->getUser()['role'] ?? '';

        // Admins see all businesses in the tenant
        if (in_array($userRole, ['admin', 'super_admin'])) {
            $stmt = $this->db->prepare(
                "SELECT c.*, 
                        (SELECT COUNT(*) FROM users u WHERE u.company_id = c.id AND u.is_active = TRUE) as user_count
                 FROM companies c 
                 WHERE c.tenant_id = :tid 
                 ORDER BY c.name"
            );
            $stmt->execute(['tid' => $tenantId]);
        } else {
            // Non-admins see only businesses they are assigned to
            $stmt = $this->db->prepare(
                "SELECT c.*, ub.role as business_role, ub.is_default,
                        (SELECT COUNT(*) FROM users u WHERE u.company_id = c.id AND u.is_active = TRUE) as user_count
                 FROM companies c
                 JOIN user_businesses ub ON c.id = ub.company_id
                 WHERE c.tenant_id = :tid AND ub.user_id = :uid
                 ORDER BY c.name"
            );
            $stmt->execute(['tid' => $tenantId, 'uid' => $userId]);
        }

        Response::success($stmt->fetchAll());
    }

    /**
     * POST /api/businesses
     * Create a new business (company) within the tenant – admin only
     */
    public function create(Request $request): void
    {
        $tenantId = $request->getTenantId();
        $data = $request->validate([
            'name' => 'required|min:2',
        ]);

        // Check tenant business limit
        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM companies WHERE tenant_id = :tid");
        $countStmt->execute(['tid' => $tenantId]);
        $currentCount = (int) $countStmt->fetchColumn();

        $tenant = $request->getTenant();
        if ($tenant && $currentCount >= ($tenant['max_businesses'] ?? 999)) {
            Response::error('Business limit reached for your plan. Please upgrade.', 403);
            return;
        }

        try {
            $this->db->beginTransaction();

            $stmt = $this->db->prepare(
                "INSERT INTO companies (
                    tenant_id, name, trading_name, tin_number, business_registration,
                    vat_registered, address, city, region, digital_address,
                    phone, email, currency, industry
                ) VALUES (
                    :tid, :name, :trading_name, :tin, :reg,
                    :vat, :address, :city, :region, :da,
                    :phone, :email, :currency, :industry
                ) RETURNING *"
            );
            $stmt->execute([
                'tid' => $tenantId,
                'name' => $data['name'],
                'trading_name' => $request->input('trading_name'),
                'tin' => $request->input('tin_number'),
                'reg' => $request->input('business_registration'),
                'vat' => $request->input('vat_registered', false) ? 'true' : 'false',
                'address' => $request->input('address'),
                'city' => $request->input('city', 'Accra'),
                'region' => $request->input('region', 'Greater Accra'),
                'da' => $request->input('digital_address'),
                'phone' => $request->input('phone'),
                'email' => $request->input('email'),
                'currency' => $request->input('currency', 'GHS'),
                'industry' => $request->input('industry'),
            ]);
            $business = $stmt->fetch();
            $companyId = $business['id'];

            // Create default chart of accounts for the new business
            $this->db->prepare("SELECT create_default_chart_of_accounts(:company_id)")
                ->execute(['company_id' => $companyId]);

            // Create default tax rates for the new business
            $this->db->prepare(
                "INSERT INTO tax_rates (company_id, tax_name, tax_code, rate, tax_type, effective_date, description)
                 SELECT :company_id, tax_name, tax_code, rate, tax_type, effective_date, description
                 FROM tax_rates WHERE company_id IS NULL"
            )->execute(['company_id' => $companyId]);

            // Auto-assign the creating user to this business
            $this->db->prepare(
                "INSERT INTO user_businesses (user_id, company_id, role, is_default) 
                 VALUES (:uid, :cid, 'admin', FALSE)
                 ON CONFLICT (user_id, company_id) DO NOTHING"
            )->execute([
                'uid' => $request->getUserId(),
                'cid' => $companyId
            ]);

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'create', 'business', $companyId);
            Response::created($business, 'Business created successfully');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to create business: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /api/businesses/{id}
     * Get a single business
     */
    public function show(Request $request, string $id = ''): void
    {
        $id = $id ?: $request->param('id');
        $tenantId = $request->getTenantId();
        $stmt = $this->db->prepare("SELECT * FROM companies WHERE id = :id AND tenant_id = :tid");
        $stmt->execute(['id' => $id, 'tid' => $tenantId]);
        $business = $stmt->fetch();

        if (!$business) {
            Response::notFound('Business not found');
            return;
        }

        // Add assigned users
        $usersStmt = $this->db->prepare(
            "SELECT u.id, u.email, u.first_name, u.last_name, u.role as global_role, 
                    ub.role as business_role, ub.is_default, u.is_active
             FROM user_businesses ub
             JOIN users u ON ub.user_id = u.id
             WHERE ub.company_id = :cid
             ORDER BY u.first_name"
        );
        $usersStmt->execute(['cid' => $id]);
        $business['assigned_users'] = $usersStmt->fetchAll();

        Response::success($business);
    }

    /**
     * PUT /api/businesses/{id}
     * Update a business
     */
    public function update(Request $request, string $id = ''): void
    {
        $id = $id ?: $request->param('id');
        $tenantId = $request->getTenantId();
        $data = $request->validate([
            'name' => 'required|min:2'
        ]);

        $stmt = $this->db->prepare(
            "UPDATE companies SET 
                name = :name, trading_name = :trading_name, tin_number = :tin,
                business_registration = :reg, vat_registered = :vat,
                address = :address, city = :city, region = :region,
                digital_address = :da, phone = :phone, email = :email,
                currency = :currency, industry = :industry,
                is_active = :active, updated_at = NOW()
             WHERE id = :id AND tenant_id = :tid RETURNING *"
        );
        $stmt->execute([
            'name' => $data['name'],
            'trading_name' => $request->input('trading_name'),
            'tin' => $request->input('tin_number'),
            'reg' => $request->input('business_registration'),
            'vat' => $request->input('vat_registered', false) ? 'true' : 'false',
            'address' => $request->input('address'),
            'city' => $request->input('city'),
            'region' => $request->input('region'),
            'da' => $request->input('digital_address'),
            'phone' => $request->input('phone'),
            'email' => $request->input('email'),
            'currency' => $request->input('currency', 'GHS'),
            'industry' => $request->input('industry'),
            'active' => ($request->input('is_active', true)) ? 'true' : 'false',
            'id' => $id,
            'tid' => $tenantId
        ]);

        $business = $stmt->fetch();
        if (!$business) {
            Response::notFound('Business not found');
            return;
        }

        $this->auditLog($id, $request->getUserId(), 'update', 'business', $id);
        Response::success($business, 'Business updated');
    }

    /**
     * DELETE /api/businesses/{id}
     * Deactivate a business (soft delete)
     */
    public function deactivate(Request $request, string $id = ''): void
    {
        $id = $id ?: $request->param('id');
        $tenantId = $request->getTenantId();

        // Don't deactivate the current business
        if ($id === $request->getCompanyId()) {
            Response::error('Cannot deactivate the business you are currently using', 400);
            return;
        }

        $stmt = $this->db->prepare(
            "UPDATE companies SET is_active = FALSE, updated_at = NOW() WHERE id = :id AND tenant_id = :tid"
        );
        $stmt->execute(['id' => $id, 'tid' => $tenantId]);

        if ($stmt->rowCount() === 0) {
            Response::notFound('Business not found');
            return;
        }

        $this->auditLog($id, $request->getUserId(), 'deactivate', 'business', $id);
        Response::success(null, 'Business deactivated');
    }

    /**
     * POST /api/businesses/{id}/assign-user
     * Assign a user to a business
     */
    public function assignUser(Request $request, string $id = ''): void
    {
        $id = $id ?: $request->param('id');
        $tenantId = $request->getTenantId();

        $data = $request->validate([
            'user_id' => 'required',
            'role' => 'required|in:admin,manager,accountant,cashier,staff,viewer'
        ]);

        // Verify business belongs to tenant
        $bizStmt = $this->db->prepare("SELECT id FROM companies WHERE id = :id AND tenant_id = :tid");
        $bizStmt->execute(['id' => $id, 'tid' => $tenantId]);
        if (!$bizStmt->fetch()) {
            Response::notFound('Business not found');
            return;
        }

        // Verify user belongs to same tenant
        $userStmt = $this->db->prepare("SELECT id FROM users WHERE id = :uid AND tenant_id = :tid");
        $userStmt->execute(['uid' => $data['user_id'], 'tid' => $tenantId]);
        if (!$userStmt->fetch()) {
            Response::notFound('User not found in this tenant');
            return;
        }

        // Upsert
        $stmt = $this->db->prepare(
            "INSERT INTO user_businesses (user_id, company_id, role)
             VALUES (:uid, :cid, :role)
             ON CONFLICT (user_id, company_id) 
             DO UPDATE SET role = :role2
             RETURNING *"
        );
        $stmt->execute([
            'uid' => $data['user_id'],
            'cid' => $id,
            'role' => $data['role'],
            'role2' => $data['role']
        ]);

        $this->auditLog($id, $request->getUserId(), 'assign_user', 'user_businesses', $data['user_id']);
        Response::success($stmt->fetch(), 'User assigned to business');
    }

    /**
     * DELETE /api/businesses/{id}/remove-user/{userId}
     * Remove a user from a business
     */
    public function removeUser(Request $request, string $id = ''): void
    {
        $id = $id ?: $request->param('id');
        $tenantId = $request->getTenantId();
        $userId = $request->input('user_id');

        if (!$userId) {
            Response::error('user_id is required', 400);
            return;
        }

        // Don't let admin remove themselves
        if ($userId === $request->getUserId()) {
            Response::error('Cannot remove yourself from a business', 400);
            return;
        }

        $stmt = $this->db->prepare(
            "DELETE FROM user_businesses 
             WHERE user_id = :uid AND company_id = :cid
             AND EXISTS (SELECT 1 FROM companies WHERE id = :cid2 AND tenant_id = :tid)"
        );
        $stmt->execute([
            'uid' => $userId,
            'cid' => $id,
            'cid2' => $id,
            'tid' => $tenantId
        ]);

        if ($stmt->rowCount() === 0) {
            Response::notFound('Assignment not found');
            return;
        }

        $this->auditLog($id, $request->getUserId(), 'remove_user', 'user_businesses', $userId);
        Response::success(null, 'User removed from business');
    }

    /**
     * POST /api/businesses/switch
     * Switch the current user's active business context
     * Returns new JWT tokens scoped to the selected business
     */
    public function switchBusiness(Request $request): void
    {
        $data = $request->validate([
            'company_id' => 'required'
        ]);

        $tenantId = $request->getTenantId();
        $userId = $request->getUserId();
        $userRole = $request->getUser()['role'] ?? '';

        // Verify the business belongs to the tenant
        $bizStmt = $this->db->prepare(
            "SELECT c.id, c.name, c.currency 
             FROM companies c 
             WHERE c.id = :cid AND c.tenant_id = :tid AND c.is_active = TRUE"
        );
        $bizStmt->execute(['cid' => $data['company_id'], 'tid' => $tenantId]);
        $business = $bizStmt->fetch();

        if (!$business) {
            Response::notFound('Business not found or inactive');
            return;
        }

        // Check user access (admins can access any business in their tenant)
        if (!in_array($userRole, ['admin', 'super_admin'])) {
            $accessStmt = $this->db->prepare(
                "SELECT role FROM user_businesses WHERE user_id = :uid AND company_id = :cid"
            );
            $accessStmt->execute(['uid' => $userId, 'cid' => $data['company_id']]);
            if (!$accessStmt->fetch()) {
                Response::error('You do not have access to this business', 403);
                return;
            }
        }

        // Update user's current company_id
        $this->db->prepare("UPDATE users SET company_id = :cid, updated_at = NOW() WHERE id = :uid")
            ->execute(['cid' => $data['company_id'], 'uid' => $userId]);

        // Fetch updated user info
        $userStmt = $this->db->prepare(
            "SELECT u.id, u.email, u.first_name, u.last_name, u.role, u.tenant_id,
                    c.id as company_id, c.name as company_name, c.currency
             FROM users u JOIN companies c ON u.company_id = c.id
             WHERE u.id = :uid"
        );
        $userStmt->execute(['uid' => $userId]);
        $user = $userStmt->fetch();

        // Generate new JWT tokens with updated company_id
        $authController = new AuthController();
        $token = $authController->generateTokenPublic(
            $user['id'], $user['company_id'], $user['tenant_id'],
            $user['email'], $user['role'], $user['first_name'], $user['last_name']
        );
        $refreshToken = $authController->generateRefreshTokenPublic($user['id'], $user['company_id'], $user['tenant_id']);

        $this->auditLog($data['company_id'], $userId, 'switch_business', 'companies', $data['company_id']);

        Response::success([
            'token' => $token,
            'refresh_token' => $refreshToken,
            'expires_in' => (int) ($_ENV['JWT_EXPIRY'] ?? 3600),
            'user' => [
                'id' => $user['id'],
                'email' => $user['email'],
                'first_name' => $user['first_name'],
                'last_name' => $user['last_name'],
                'role' => $user['role'],
                'tenant_id' => $user['tenant_id'],
                'company' => [
                    'id' => $user['company_id'],
                    'name' => $user['company_name'],
                    'currency' => $user['currency']
                ]
            ]
        ], 'Switched business successfully');
    }
}
