<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class AdminController extends Controller
{
    /**
     * GET /admin/users
     */
    public function listUsers(Request $request): void
    {
        $tenantId = $request->getTenantId();
        $stmt = $this->db->prepare(
            "SELECT id, company_id, email, first_name, last_name, role, is_active, last_login, created_at
             FROM users WHERE tenant_id = :tid ORDER BY first_name"
        );
        $stmt->execute(['tid' => $tenantId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /admin/users
     */
    public function createUser(Request $request): void
    {
        $data = $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:8',
            'first_name' => 'required',
            'last_name' => 'required',
            'role' => 'required|in:admin,accountant,cashier,manager,staff,viewer'
        ]);

        $companyId = $request->getCompanyId();

        // Check duplicate email
        $check = $this->db->prepare("SELECT id FROM users WHERE email = :email");
        $check->execute(['email' => $data['email']]);
        if ($check->fetch()) {
            Response::error('Email already in use', 409);
            return;
        }

        $stmt = $this->db->prepare(
            "INSERT INTO users (company_id, tenant_id, email, password_hash, first_name, last_name, role)
             VALUES (:cid, :tid, :email, :pw, :fn, :ln, :role) RETURNING id, email, first_name, last_name, role, created_at"
        );
        $stmt->execute([
            'cid' => $companyId,
            'tid' => $request->getTenantId(),
            'email' => $data['email'],
            'pw' => password_hash($data['password'], PASSWORD_BCRYPT, ['cost' => 12]),
            'fn' => $data['first_name'],
            'ln' => $data['last_name'],
            'role' => $data['role']
        ]);

        $newUser = $stmt->fetch();
        if ($newUser) {
            // Auto-assign user to the current business
            $this->db->prepare(
                "INSERT INTO user_businesses (user_id, company_id, role, is_default) VALUES (:uid, :cid, :role, TRUE)
                 ON CONFLICT (user_id, company_id) DO NOTHING"
            )->execute(['uid' => $newUser['id'], 'cid' => $companyId, 'role' => $data['role']]);
        }

        $this->auditLog($companyId, $request->getUserId(), 'user_created', 'users', $newUser['id'] ?? null);
        Response::created($newUser, 'User created');
    }

    /**
     * PUT /admin/users/{id}
     */
    public function updateUser(Request $request, string $id): void
    {
        $companyId = $request->getCompanyId();
        $data = $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'role' => 'required|in:admin,accountant,cashier,manager,staff,viewer'
        ]);

        $isActive = (bool) $request->input('is_active', true);

        $stmt = $this->db->prepare(
            "UPDATE users SET first_name = :fn, last_name = :ln, role = :role, is_active = :active, updated_at = NOW()
             WHERE id = :id AND company_id = :cid RETURNING id, email, first_name, last_name, role, is_active"
        );
        $stmt->execute([
            'fn' => $data['first_name'],
            'ln' => $data['last_name'],
            'role' => $data['role'],
            'active' => $isActive ? 'true' : 'false',
            'id' => $id,
            'cid' => $companyId
        ]);

        $user = $stmt->fetch();
        if (!$user) {
            Response::notFound('User not found');
            return;
        }
        $this->auditLog($companyId, $request->getUserId(), 'user_updated', 'users', $id);
        Response::success($user, 'User updated');
    }

    /**
     * DELETE /admin/users/{id}
     */
    public function deactivateUser(Request $request, string $id): void
    {
        $companyId = $request->getCompanyId();

        // Don't deactivate self
        if ($id === $request->getUserId()) {
            Response::error('Cannot deactivate your own account', 400);
            return;
        }

        $stmt = $this->db->prepare(
            "UPDATE users SET is_active = FALSE, updated_at = NOW() WHERE id = :id AND company_id = :cid"
        );
        $stmt->execute(['id' => $id, 'cid' => $companyId]);

        if ($stmt->rowCount() === 0) {
            Response::notFound('User not found');
            return;
        }
        $this->auditLog($companyId, $request->getUserId(), 'user_deactivated', 'users', $id);
        Response::success(null, 'User deactivated');
    }

    /**
     * GET /admin/company
     */
    public function getCompany(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare("SELECT * FROM companies WHERE id = :id");
        $stmt->execute(['id' => $companyId]);
        $company = $stmt->fetch();
        if (!$company) {
            Response::notFound('Company not found');
            return;
        }
        Response::success($company);
    }

    /**
     * PUT /admin/company
     */
    public function updateCompany(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $data = $request->validate([
            'name' => 'required',
            'business_type' => 'required'
        ]);

        $stmt = $this->db->prepare(
            "UPDATE companies SET
                name = :name,
                business_type = :bt,
                tin_number = :tin,
                phone = :phone,
                email = :email,
                address = :address,
                city = :city,
                region = :region,
                digital_address = :da,
                logo_url = :logo,
                currency = :currency,
                supported_currencies = :sc,
                fiscal_year_start = :fys,
                default_profit_margin = :dpm,
                updated_at = NOW()
             WHERE id = :id RETURNING *"
        );
        $stmt->execute([
            'name' => $data['name'],
            'bt' => $data['business_type'],
            'tin' => $request->input('tin_number'),
            'phone' => $request->input('phone'),
            'email' => $request->input('email'),
            'address' => $request->input('address'),
            'city' => $request->input('city'),
            'region' => $request->input('region'),
            'da' => $request->input('digital_address'),
            'logo' => $request->input('logo_url'),
            'currency' => $request->input('currency', 'GHS'),
            'sc' => $request->input('supported_currencies', ''),
            'fys' => $request->input('fiscal_year_start', 1),
            'dpm' => $request->input('default_profit_margin', 0),
            'id' => $companyId
        ]);
        $this->auditLog($companyId, $request->getUserId(), 'company_updated', 'companies', $companyId);
        Response::success($stmt->fetch(), 'Company updated');
    }

    /**
     * GET /admin/audit-log
     */
    public function auditLogList(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $page = (int) $request->query('page', 1);
        $limit = min((int) $request->query('limit', 50), 100);
        $offset = ($page - 1) * $limit;

        $where = "a.company_id = :cid";
        $params = ['cid' => $companyId];

        $userId = $request->query('user_id');
        if ($userId) {
            $where .= " AND a.user_id = :uid";
            $params['uid'] = $userId;
        }

        $action = $request->query('action');
        if ($action) {
            $where .= " AND a.action = :action";
            $params['action'] = $action;
        }

        $dateFrom = $request->query('date_from');
        if ($dateFrom) {
            $where .= " AND a.created_at >= :datefrom";
            $params['datefrom'] = $dateFrom;
        }

        $dateTo = $request->query('date_to');
        if ($dateTo) {
            $where .= " AND a.created_at <= :dateto";
            $params['dateto'] = $dateTo . ' 23:59:59';
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM audit_log a WHERE $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $stmt = $this->db->prepare(
            "SELECT a.*, u.first_name || ' ' || u.last_name as user_name, u.email as user_email
             FROM audit_log a LEFT JOIN users u ON a.user_id = u.id
             WHERE $where ORDER BY a.created_at DESC LIMIT :lim OFFSET :off"
        );
        foreach ($params as $k => $v) $stmt->bindValue($k, $v);
        $stmt->bindValue('lim', $limit, \PDO::PARAM_INT);
        $stmt->bindValue('off', $offset, \PDO::PARAM_INT);
        $stmt->execute();

        Response::paginated($stmt->fetchAll(), $total, $page, $limit);
    }

    /**
     * GET /admin/bank-accounts
     */
    public function listBankAccounts(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $stmt = $this->db->prepare(
            "SELECT ba.*, coa.account_name as ledger_account_name
             FROM bank_accounts ba
             LEFT JOIN chart_of_accounts coa ON ba.account_id = coa.id
             WHERE ba.company_id = :cid ORDER BY ba.bank_name"
        );
        $stmt->execute(['cid' => $companyId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * POST /admin/bank-accounts
     */
    public function createBankAccount(Request $request): void
    {
        $data = $request->validate([
            'bank_name' => 'required',
            'account_name' => 'required',
            'account_number' => 'required',
            'account_type' => 'required|in:savings,current,mobile_money'
        ]);

        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "INSERT INTO bank_accounts (company_id, bank_name, account_name, account_number, account_type, branch, opening_balance, current_balance, account_id)
             VALUES (:cid, :bank, :name, :num, :type, :branch, :ob, :ob2, :linked) RETURNING *"
        );
        $openingBalance = (float) $request->input('opening_balance', 0);
        $stmt->execute([
            'cid' => $companyId,
            'bank' => $data['bank_name'],
            'name' => $data['account_name'],
            'num' => $data['account_number'],
            'type' => $data['account_type'],
            'branch' => $request->input('branch'),
            'ob' => $openingBalance,
            'ob2' => $openingBalance,
            'linked' => $request->input('account_id')
        ]);

        Response::created($stmt->fetch(), 'Bank account added');
    }

    /**
     * GET /admin/notifications
     */
    public function getNotifications(Request $request): void
    {
        $userId = $request->getUserId();
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT * FROM notifications WHERE company_id = :cid
             AND (user_id = :uid OR user_id IS NULL)
             ORDER BY created_at DESC LIMIT 50"
        );
        $stmt->execute(['cid' => $companyId, 'uid' => $userId]);
        Response::success($stmt->fetchAll());
    }

    /**
     * PUT /admin/notifications/{id}/read
     */
    public function markNotificationRead(Request $request, string $id): void
    {
        $stmt = $this->db->prepare("UPDATE notifications SET is_read = TRUE WHERE id = :id");
        $stmt->execute(['id' => $id]);
        Response::success(null, 'Notification marked as read');
    }

    /**
     * GET /admin/backup
     * Returns a JSON export of all company data
     */
    public function exportData(Request $request): void
    {
        $companyId = $request->getCompanyId();

        $tables = [
            'chart_of_accounts', 'contacts', 'products', 'product_categories',
            'journal_entries', 'journal_entry_lines', 'invoices', 'invoice_items',
            'payments', 'bank_accounts', 'employees', 'payroll_periods', 'payslips',
            'tax_filings', 'fixed_assets'
        ];

        $backup = ['company_id' => $companyId, 'exported_at' => date('c'), 'data' => []];

        foreach ($tables as $table) {
            $stmt = $this->db->prepare("SELECT * FROM $table WHERE company_id = :cid");
            $stmt->execute(['cid' => $companyId]);
            $backup['data'][$table] = $stmt->fetchAll();
        }

        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="cashbook-backup-' . date('Y-m-d') . '.json"');
        echo json_encode($backup, JSON_PRETTY_PRINT);
        exit;
    }
}
