<?php

namespace CashBook\Controllers;

use CashBook\Core\Controller;
use CashBook\Core\Request;
use CashBook\Core\Response;

class AccountingController extends Controller
{
    /**
     * GET /accounting/chart-of-accounts
     */
    public function getChartOfAccounts(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $type = $request->query('type');

        $sql = "SELECT * FROM chart_of_accounts WHERE company_id = :company_id AND is_active = TRUE";
        $params = ['company_id' => $companyId];

        if ($type) {
            $sql .= " AND account_type = :type";
            $params['type'] = $type;
        }
        $sql .= " ORDER BY account_code";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $accounts = $stmt->fetchAll();

        Response::success($accounts);
    }

    /**
     * POST /accounting/chart-of-accounts
     */
    public function createAccount(Request $request): void
    {
        $data = $request->validate([
            'account_code' => 'required',
            'account_name' => 'required',
            'account_type' => 'required|in:asset,liability,equity,revenue,expense'
        ]);

        $companyId = $request->getCompanyId();

        // Check unique code
        $check = $this->db->prepare("SELECT id FROM chart_of_accounts WHERE company_id = :cid AND account_code = :code");
        $check->execute(['cid' => $companyId, 'code' => $data['account_code']]);
        if ($check->fetch()) {
            Response::error('Account code already exists', 409);
            return;
        }

        $normalBalance = in_array($data['account_type'], ['asset', 'expense']) ? 'debit' : 'credit';

        $stmt = $this->db->prepare(
            "INSERT INTO chart_of_accounts (company_id, account_code, account_name, account_type, sub_type, description, normal_balance, parent_account_id, opening_balance)
             VALUES (:company_id, :account_code, :account_name, :account_type, :sub_type, :description, :normal_balance, :parent_account_id, :opening_balance)
             RETURNING *"
        );
        $stmt->execute([
            'company_id' => $companyId,
            'account_code' => $data['account_code'],
            'account_name' => $data['account_name'],
            'account_type' => $data['account_type'],
            'sub_type' => $request->input('sub_type'),
            'description' => $request->input('description'),
            'normal_balance' => $normalBalance,
            'parent_account_id' => $request->input('parent_account_id'),
            'opening_balance' => $request->input('opening_balance', 0)
        ]);

        $account = $stmt->fetch();
        $this->auditLog($companyId, $request->getUserId(), 'create', 'chart_of_accounts', $account['id'], null, $account);
        Response::created($account);
    }

    /**
     * POST /accounting/journal-entries
     */
    public function createJournalEntry(Request $request): void
    {
        $data = $request->validate([
            'entry_date' => 'required',
            'description' => 'required'
        ]);

        $lines = $request->input('lines', []);
        if (count($lines) < 2) {
            Response::error('Journal entry must have at least 2 lines', 422);
            return;
        }

        $totalDebit = 0;
        $totalCredit = 0;
        foreach ($lines as $line) {
            $totalDebit += (float) ($line['debit_amount'] ?? 0);
            $totalCredit += (float) ($line['credit_amount'] ?? 0);
        }

        if (abs($totalDebit - $totalCredit) > 0.01) {
            Response::error('Debits and credits must be equal. Debit: ' . $totalDebit . ', Credit: ' . $totalCredit, 422);
            return;
        }

        $companyId = $request->getCompanyId();

        try {
            $this->db->beginTransaction();

            $entryNumber = $this->generateNumber('JE', 'journal_entries', 'entry_number', $companyId);

            $stmt = $this->db->prepare(
                "INSERT INTO journal_entries (company_id, entry_number, entry_date, description, reference, source, total_debit, total_credit, status, created_by)
                 VALUES (:company_id, :entry_number, :entry_date, :description, :reference, :source, :total_debit, :total_credit, :status, :created_by)
                 RETURNING *"
            );
            $stmt->execute([
                'company_id' => $companyId,
                'entry_number' => $entryNumber,
                'entry_date' => $data['entry_date'],
                'description' => $data['description'],
                'reference' => $request->input('reference'),
                'source' => $request->input('source', 'manual'),
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'status' => $request->input('status', 'draft'),
                'created_by' => $request->getUserId()
            ]);
            $entry = $stmt->fetch();

            // Insert lines
            $lineStmt = $this->db->prepare(
                "INSERT INTO journal_entry_lines (journal_entry_id, account_id, debit_amount, credit_amount, description, tax_code, tax_amount)
                 VALUES (:journal_entry_id, :account_id, :debit_amount, :credit_amount, :description, :tax_code, :tax_amount)"
            );

            foreach ($lines as $line) {
                $lineStmt->execute([
                    'journal_entry_id' => $entry['id'],
                    'account_id' => $line['account_id'],
                    'debit_amount' => $line['debit_amount'] ?? 0,
                    'credit_amount' => $line['credit_amount'] ?? 0,
                    'description' => $line['description'] ?? null,
                    'tax_code' => $line['tax_code'] ?? null,
                    'tax_amount' => $line['tax_amount'] ?? 0
                ]);
            }

            // If posting immediately, update account balances
            if ($request->input('status') === 'posted') {
                $this->postJournalEntry($entry['id']);
            }

            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'create', 'journal_entry', $entry['id'], null, $entry);
            Response::created($entry, 'Journal entry created');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to create journal entry: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /accounting/journal-entries/{id}/post
     */
    public function postEntry(Request $request): void
    {
        $entryId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare("SELECT * FROM journal_entries WHERE id = :id AND company_id = :company_id");
        $stmt->execute(['id' => $entryId, 'company_id' => $companyId]);
        $entry = $stmt->fetch();

        if (!$entry) {
            Response::notFound('Journal entry not found');
            return;
        }

        if ($entry['status'] === 'posted') {
            Response::error('Entry already posted', 400);
            return;
        }

        try {
            $this->db->beginTransaction();
            $this->postJournalEntry($entryId);
            $this->db->prepare("UPDATE journal_entries SET status = 'posted', posted_at = NOW(), approved_by = :user_id WHERE id = :id")
                ->execute(['id' => $entryId, 'user_id' => $request->getUserId()]);
            $this->db->commit();

            $this->auditLog($companyId, $request->getUserId(), 'post', 'journal_entry', $entryId);
            Response::success(null, 'Journal entry posted');
        } catch (\Exception $e) {
            $this->db->rollBack();
            Response::error('Failed to post entry: ' . $e->getMessage(), 500);
        }
    }

    /**
     * GET /accounting/journal-entries
     */
    public function getJournalEntries(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $pagination = $this->paginate($request);

        $where = "WHERE je.company_id = :company_id";
        $params = ['company_id' => $companyId];

        if ($request->query('status')) {
            $where .= " AND je.status = :status";
            $params['status'] = $request->query('status');
        }
        if ($request->query('date_from')) {
            $where .= " AND je.entry_date >= :date_from";
            $params['date_from'] = $request->query('date_from');
        }
        if ($request->query('date_to')) {
            $where .= " AND je.entry_date <= :date_to";
            $params['date_to'] = $request->query('date_to');
        }

        $countStmt = $this->db->prepare("SELECT COUNT(*) FROM journal_entries je $where");
        $countStmt->execute($params);
        $total = (int) $countStmt->fetchColumn();

        $sql = "SELECT je.*, u.first_name || ' ' || u.last_name as created_by_name
                FROM journal_entries je
                LEFT JOIN users u ON je.created_by = u.id
                $where ORDER BY je.entry_date DESC, je.created_at DESC
                LIMIT {$pagination['per_page']} OFFSET {$pagination['offset']}";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $entries = $stmt->fetchAll();

        Response::paginated($entries, $total, $pagination['page'], $pagination['per_page']);
    }

    /**
     * GET /accounting/journal-entries/{id}
     */
    public function getJournalEntry(Request $request): void
    {
        $entryId = $request->param('id');
        $companyId = $request->getCompanyId();

        $stmt = $this->db->prepare(
            "SELECT je.*, u.first_name || ' ' || u.last_name as created_by_name
             FROM journal_entries je LEFT JOIN users u ON je.created_by = u.id
             WHERE je.id = :id AND je.company_id = :company_id"
        );
        $stmt->execute(['id' => $entryId, 'company_id' => $companyId]);
        $entry = $stmt->fetch();

        if (!$entry) {
            Response::notFound('Journal entry not found');
            return;
        }

        // Get lines
        $lineStmt = $this->db->prepare(
            "SELECT jel.*, coa.account_code, coa.account_name, coa.account_type
             FROM journal_entry_lines jel
             JOIN chart_of_accounts coa ON jel.account_id = coa.id
             WHERE jel.journal_entry_id = :entry_id
             ORDER BY jel.debit_amount DESC"
        );
        $lineStmt->execute(['entry_id' => $entryId]);
        $entry['lines'] = $lineStmt->fetchAll();

        Response::success($entry);
    }

    /**
     * GET /accounting/trial-balance
     */
    public function trialBalance(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $asOf = $request->query('as_of', date('Y-m-d'));

        $stmt = $this->db->prepare(
            "SELECT coa.account_code, coa.account_name, coa.account_type, coa.normal_balance,
                    COALESCE(SUM(jel.debit_amount), 0) as total_debit,
                    COALESCE(SUM(jel.credit_amount), 0) as total_credit,
                    CASE WHEN coa.normal_balance = 'debit' 
                         THEN COALESCE(SUM(jel.debit_amount), 0) - COALESCE(SUM(jel.credit_amount), 0)
                         ELSE COALESCE(SUM(jel.credit_amount), 0) - COALESCE(SUM(jel.debit_amount), 0)
                    END as balance
             FROM chart_of_accounts coa
             LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
             LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.entry_date <= :as_of
             WHERE coa.company_id = :company_id AND coa.is_active = TRUE
             GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.normal_balance
             HAVING COALESCE(SUM(jel.debit_amount), 0) != 0 OR COALESCE(SUM(jel.credit_amount), 0) != 0
             ORDER BY coa.account_code"
        );
        $stmt->execute(['company_id' => $companyId, 'as_of' => $asOf]);
        $accounts = $stmt->fetchAll();

        $totalDebit = array_sum(array_column($accounts, 'total_debit'));
        $totalCredit = array_sum(array_column($accounts, 'total_credit'));

        Response::success([
            'as_of' => $asOf,
            'accounts' => $accounts,
            'total_debit' => $totalDebit,
            'total_credit' => $totalCredit,
            'is_balanced' => abs($totalDebit - $totalCredit) < 0.01
        ]);
    }

    /**
     * GET /accounting/general-ledger
     */
    public function generalLedger(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $accountId = $request->query('account_id');
        $dateFrom = $request->query('date_from', date('Y-01-01'));
        $dateTo = $request->query('date_to', date('Y-m-d'));

        if (!$accountId) {
            Response::error('account_id is required', 422);
            return;
        }

        $stmt = $this->db->prepare(
            "SELECT jel.*, je.entry_number, je.entry_date, je.description as entry_description, je.reference, je.source,
                    coa.account_code, coa.account_name
             FROM journal_entry_lines jel
             JOIN journal_entries je ON jel.journal_entry_id = je.id
             JOIN chart_of_accounts coa ON jel.account_id = coa.id
             WHERE je.company_id = :company_id AND je.status = 'posted'
               AND jel.account_id = :account_id
               AND je.entry_date BETWEEN :date_from AND :date_to
             ORDER BY je.entry_date, je.created_at"
        );
        $stmt->execute([
            'company_id' => $companyId,
            'account_id' => $accountId,
            'date_from' => $dateFrom,
            'date_to' => $dateTo
        ]);
        $ledger = $stmt->fetchAll();

        // Calculate running balance
        $balance = 0;
        foreach ($ledger as &$entry) {
            $balance += $entry['debit_amount'] - $entry['credit_amount'];
            $entry['running_balance'] = $balance;
        }

        Response::success([
            'account_id' => $accountId,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
            'entries' => $ledger,
            'closing_balance' => $balance
        ]);
    }

    /**
     * Update account balances when posting a journal entry
     */
    private function postJournalEntry(string $entryId): void
    {
        $lines = $this->db->prepare(
            "SELECT account_id, debit_amount, credit_amount FROM journal_entry_lines WHERE journal_entry_id = :id"
        );
        $lines->execute(['id' => $entryId]);

        $updateStmt = $this->db->prepare(
            "UPDATE chart_of_accounts SET 
                current_balance = current_balance + 
                    CASE WHEN normal_balance = 'debit' THEN :debit - :credit ELSE :credit2 - :debit2 END,
                updated_at = NOW()
             WHERE id = :account_id"
        );

        foreach ($lines->fetchAll() as $line) {
            $updateStmt->execute([
                'debit' => $line['debit_amount'],
                'credit' => $line['credit_amount'],
                'credit2' => $line['credit_amount'],
                'debit2' => $line['debit_amount'],
                'account_id' => $line['account_id']
            ]);
        }
    }

    /**
     * Update an existing chart of accounts entry
     */
    public function updateAccount(Request $request): void
    {
        $companyId = $request->getCompanyId();
        $id = $request->param('id');

        // Check account exists
        $stmt = $this->db->prepare("SELECT * FROM chart_of_accounts WHERE id = :id AND company_id = :cid");
        $stmt->execute(['id' => $id, 'cid' => $companyId]);
        $existing = $stmt->fetch();

        if (!$existing) {
            Response::notFound('Account not found');
            return;
        }

        $allowedFields = [
            'account_name', 'account_type', 'parent_account_id',
            'description', 'is_active', 'tax_rate_id'
        ];

        $updates = [];
        $params = ['id' => $id, 'cid' => $companyId];

        foreach ($allowedFields as $field) {
            $value = $request->input($field);
            if ($value !== null) {
                $updates[] = "$field = :$field";
                $params[$field] = $value;
            }
        }

        if (empty($updates)) {
            Response::error('No fields to update', 400);
            return;
        }

        $updates[] = "updated_at = NOW()";
        $sql = "UPDATE chart_of_accounts SET " . implode(', ', $updates)
             . " WHERE id = :id AND company_id = :cid RETURNING *";
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        $account = $stmt->fetch();

        $this->auditLog($companyId, $request->getUserId(), 'update', 'chart_of_accounts', $id, $existing, $account);

        Response::success($account, 'Account updated successfully');
    }
}
