<?php
/**
 * CashBook ERP - API Routes
 * All routes for the Ghana SME bookkeeping application
 *
 * $router is injected from public/index.php
 */

use CashBook\Core\Router;

// ─────────────────────────────────────────────
// Public Routes (no authentication required)
// ─────────────────────────────────────────────
$router->post('/api/auth/register', ['CashBook\Controllers\AuthController', 'register']);
$router->post('/api/auth/login', ['CashBook\Controllers\AuthController', 'login']);
$router->post('/api/auth/refresh', ['CashBook\Controllers\AuthController', 'refresh']);

// Health check
$router->get('/api/health', function () {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'ok',
        'app' => 'CashBook ERP',
        'version' => '1.0.0',
        'timestamp' => date('c'),
        'timezone' => 'Africa/Accra'
    ]);
});

// ─────────────────────────────────────────────
// Authenticated Routes (JWT required)
// ─────────────────────────────────────────────
$router->group(['middleware' => ['CashBook\Middleware\AuthMiddleware', 'CashBook\Middleware\TenantMiddleware']], function (Router $router) {

    // --- Auth / Profile ---
    $router->get('/api/auth/profile', ['CashBook\Controllers\AuthController', 'me']);
    $router->put('/api/auth/password', ['CashBook\Controllers\AuthController', 'changePassword']);

    // ─── Tenant Management (admin only) ───
    $router->get('/api/tenant', ['CashBook\Controllers\TenantController', 'show']);
    $router->put('/api/tenant', ['CashBook\Controllers\TenantController', 'update']);

    // ─── Business Management ───
    $router->get('/api/businesses', ['CashBook\Controllers\BusinessController', 'index']);
    $router->post('/api/businesses', ['CashBook\Controllers\BusinessController', 'create']);
    $router->get('/api/businesses/{id}', ['CashBook\Controllers\BusinessController', 'show']);
    $router->put('/api/businesses/{id}', ['CashBook\Controllers\BusinessController', 'update']);
    $router->delete('/api/businesses/{id}', ['CashBook\Controllers\BusinessController', 'deactivate']);
    $router->post('/api/businesses/{id}/assign-user', ['CashBook\Controllers\BusinessController', 'assignUser']);
    $router->post('/api/businesses/{id}/remove-user', ['CashBook\Controllers\BusinessController', 'removeUser']);
    $router->post('/api/businesses/switch', ['CashBook\Controllers\BusinessController', 'switchBusiness']);

    // ─── Accounting ───
    $router->get('/api/accounts', ['CashBook\Controllers\AccountingController', 'getChartOfAccounts']);
    $router->post('/api/accounts', ['CashBook\Controllers\AccountingController', 'createAccount']);
    $router->put('/api/accounts/{id}', ['CashBook\Controllers\AccountingController', 'updateAccount']);

    $router->get('/api/journal-entries', ['CashBook\Controllers\AccountingController', 'getJournalEntries']);
    $router->post('/api/journal-entries', ['CashBook\Controllers\AccountingController', 'createJournalEntry']);
    $router->get('/api/journal-entries/{id}', ['CashBook\Controllers\AccountingController', 'getJournalEntry']);
    $router->post('/api/journal-entries/{id}/post', ['CashBook\Controllers\AccountingController', 'postEntry']);

    $router->get('/api/trial-balance', ['CashBook\Controllers\AccountingController', 'trialBalance']);
    $router->get('/api/general-ledger', ['CashBook\Controllers\AccountingController', 'generalLedger']);

    // ─── Invoicing & Billing ───
    $router->get('/api/invoices', ['CashBook\Controllers\InvoiceController', 'index']);
    $router->post('/api/invoices', ['CashBook\Controllers\InvoiceController', 'create']);
    $router->get('/api/invoices/{id}', ['CashBook\Controllers\InvoiceController', 'show']);
    $router->put('/api/invoices/{id}', ['CashBook\Controllers\InvoiceController', 'update']);
    $router->delete('/api/invoices/{id}', ['CashBook\Controllers\InvoiceController', 'voidInvoice']);

    $router->get('/api/payments', ['CashBook\Controllers\InvoiceController', 'getPayments']);
    $router->post('/api/payments', ['CashBook\Controllers\InvoiceController', 'recordPayment']);

    // ─── Transactions (Income & Expenses) ───
    $router->get('/api/transactions', ['CashBook\Controllers\TransactionController', 'index']);
    $router->post('/api/transactions', ['CashBook\Controllers\TransactionController', 'create']);
    $router->get('/api/transactions/summary', ['CashBook\Controllers\TransactionController', 'summary']);
    $router->get('/api/transactions/{id}', ['CashBook\Controllers\TransactionController', 'show']);
    $router->delete('/api/transactions/{id}', ['CashBook\Controllers\TransactionController', 'void']);

    // ─── POS (Point of Sale) ───
    $router->get('/api/pos/terminals', ['CashBook\Controllers\POSController', 'getTerminals']);
    $router->post('/api/pos/terminals', ['CashBook\Controllers\POSController', 'createTerminal']);

    $router->post('/api/pos/sessions/open', ['CashBook\Controllers\POSController', 'openSession']);
    $router->post('/api/pos/sessions/{id}/close', ['CashBook\Controllers\POSController', 'closeSession']);
    $router->get('/api/pos/sessions/current', ['CashBook\Controllers\POSController', 'currentSession']);

    $router->get('/api/pos/sales', ['CashBook\Controllers\POSController', 'getSales']);
    $router->post('/api/pos/sales', ['CashBook\Controllers\POSController', 'createSale']);
    $router->get('/api/pos/sales/{id}', ['CashBook\Controllers\POSController', 'getSale']);
    $router->post('/api/pos/sales/{id}/refund', ['CashBook\Controllers\POSController', 'refundSale']);

    $router->get('/api/pos/returns', ['CashBook\Controllers\POSController', 'getReturns']);

    $router->get('/api/pos/daily-report', ['CashBook\Controllers\POSController', 'dailyReport']);

    // ─── Payroll ───
    $router->get('/api/employees', ['CashBook\Controllers\PayrollController', 'getEmployees']);
    $router->post('/api/employees', ['CashBook\Controllers\PayrollController', 'createEmployee']);
    $router->get('/api/employees/{id}', ['CashBook\Controllers\PayrollController', 'getEmployee']);
    $router->put('/api/employees/{id}', ['CashBook\Controllers\PayrollController', 'updateEmployee']);

    $router->post('/api/payroll/run', ['CashBook\Controllers\PayrollController', 'runPayroll']);
    $router->get('/api/payroll/periods', ['CashBook\Controllers\PayrollController', 'getPayrollPeriods']);
    $router->get('/api/payroll/periods/{id}/payslips', ['CashBook\Controllers\PayrollController', 'getPayslips']);
    $router->post('/api/payroll/periods/{id}/approve', ['CashBook\Controllers\PayrollController', 'approvePayroll']);
    $router->get('/api/payroll/tax-brackets', ['CashBook\Controllers\PayrollController', 'getTaxBrackets']);

    // ─── Inventory & Products ───
    $router->get('/api/products', ['CashBook\Controllers\InventoryController', 'index']);
    $router->post('/api/products', ['CashBook\Controllers\InventoryController', 'create']);
    $router->post('/api/products/import', ['CashBook\Controllers\InventoryController', 'import']);
    $router->get('/api/products/{id}', ['CashBook\Controllers\InventoryController', 'show']);
    $router->put('/api/products/{id}', ['CashBook\Controllers\InventoryController', 'update']);
    $router->get('/api/products/barcode/{code}', ['CashBook\Controllers\InventoryController', 'lookupBarcode']);

    $router->post('/api/products/{id}/adjust-stock', ['CashBook\Controllers\InventoryController', 'adjustStock']);
    $router->get('/api/inventory/low-stock', ['CashBook\Controllers\InventoryController', 'lowStock']);
    $router->get('/api/inventory/movements', ['CashBook\Controllers\InventoryController', 'movements']);

    $router->get('/api/product-categories', ['CashBook\Controllers\InventoryController', 'getCategories']);
    $router->post('/api/product-categories', ['CashBook\Controllers\InventoryController', 'createCategory']);
    $router->put('/api/product-categories/{id}', ['CashBook\Controllers\InventoryController', 'updateCategory']);
    $router->delete('/api/product-categories/{id}', ['CashBook\Controllers\InventoryController', 'deleteCategory']);

    // ─── Contacts (Customers & Suppliers) ───
    $router->get('/api/contacts', ['CashBook\Controllers\ContactController', 'index']);
    $router->post('/api/contacts', ['CashBook\Controllers\ContactController', 'create']);
    $router->get('/api/contacts/{id}', ['CashBook\Controllers\ContactController', 'show']);
    $router->put('/api/contacts/{id}', ['CashBook\Controllers\ContactController', 'update']);

    // ─── Fixed Assets ───
    $router->get('/api/assets', ['CashBook\Controllers\AssetController', 'index']);
    $router->post('/api/assets', ['CashBook\Controllers\AssetController', 'create']);
    $router->get('/api/assets/{id}', ['CashBook\Controllers\AssetController', 'show']);
    $router->post('/api/assets/{id}/depreciate', ['CashBook\Controllers\AssetController', 'runDepreciation']);
    $router->post('/api/assets/{id}/dispose', ['CashBook\Controllers\AssetController', 'dispose']);

    // ─── Tax Compliance ───
    $router->get('/api/tax/rates', ['CashBook\Controllers\TaxController', 'getRates']);
    $router->post('/api/tax/calculate', ['CashBook\Controllers\TaxController', 'calculateTax']);
    $router->post('/api/tax/calculate-paye', ['CashBook\Controllers\TaxController', 'calculatePAYE']);
    $router->get('/api/tax/filings', ['CashBook\Controllers\TaxController', 'getFilings']);
    $router->post('/api/tax/filings', ['CashBook\Controllers\TaxController', 'createFiling']);
    $router->get('/api/tax/withholding-summary', ['CashBook\Controllers\TaxController', 'withholdingSummary']);

    // ─── Financial Reports ───
    $router->get('/api/reports/income-statement', ['CashBook\Controllers\ReportController', 'incomeStatement']);
    $router->get('/api/reports/balance-sheet', ['CashBook\Controllers\ReportController', 'balanceSheet']);
    $router->get('/api/reports/cash-flow', ['CashBook\Controllers\ReportController', 'cashFlowStatement']);
    $router->get('/api/reports/tax-summary', ['CashBook\Controllers\ReportController', 'taxSummary']);
    $router->get('/api/reports/aging', ['CashBook\Controllers\ReportController', 'arAging']);
    $router->get('/api/reports/dashboard', ['CashBook\Controllers\ReportController', 'dashboard']);

    // ─── File Attachments ───
    $router->get('/api/attachments', ['CashBook\Controllers\FileAttachmentController', 'index']);
    $router->post('/api/attachments', ['CashBook\Controllers\FileAttachmentController', 'upload']);
    $router->get('/api/attachments/{id}/download', ['CashBook\Controllers\FileAttachmentController', 'download']);
    $router->delete('/api/attachments/{id}', ['CashBook\Controllers\FileAttachmentController', 'delete']);

    // ─── Admin / Settings (admin & manager only) ───
    $router->group(['middleware' => ['CashBook\Middleware\RoleMiddleware:admin,manager']], function (Router $router) {

        $router->get('/api/admin/users', ['CashBook\Controllers\AdminController', 'listUsers']);
        $router->post('/api/admin/users', ['CashBook\Controllers\AdminController', 'createUser']);
        $router->put('/api/admin/users/{id}', ['CashBook\Controllers\AdminController', 'updateUser']);
        $router->delete('/api/admin/users/{id}', ['CashBook\Controllers\AdminController', 'deactivateUser']);

        $router->get('/api/admin/company', ['CashBook\Controllers\AdminController', 'getCompany']);
        $router->put('/api/admin/company', ['CashBook\Controllers\AdminController', 'updateCompany']);

        $router->get('/api/admin/audit-log', ['CashBook\Controllers\AdminController', 'auditLogList']);

        $router->get('/api/admin/bank-accounts', ['CashBook\Controllers\AdminController', 'listBankAccounts']);
        $router->post('/api/admin/bank-accounts', ['CashBook\Controllers\AdminController', 'createBankAccount']);

        $router->get('/api/admin/notifications', ['CashBook\Controllers\AdminController', 'getNotifications']);
        $router->put('/api/admin/notifications/{id}/read', ['CashBook\Controllers\AdminController', 'markNotificationRead']);

        $router->get('/api/admin/backup', ['CashBook\Controllers\AdminController', 'exportData']);
    });
});
