<?php
/**
 * CashBook ERP - API Entry Point
 * Handles all API requests via routing
 */

// Bootstrap
require_once __DIR__ . '/../config/app.php';

// CORS Headers
// In production on the same domain, the browser won't send an Origin header
// for same-origin requests, so CORS is effectively a no-op. We still set
// it for any cross-origin callers (e.g. mobile apps, third-party tools).
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$allowedOrigin = '';

if (APP_ENV === 'production') {
    // Only allow the configured frontend URL (same domain in prod)
    if ($origin === FRONTEND_URL || $origin === APP_URL) {
        $allowedOrigin = $origin;
    }
} else {
    // Development: allow any localhost origin
    if (preg_match('/^https?:\/\/localhost(:\d+)?$/', $origin)) {
        $allowedOrigin = $origin;
    }
}

if ($allowedOrigin) {
    header('Access-Control-Allow-Origin: ' . $allowedOrigin);
    header('Access-Control-Allow-Credentials: true');
}
header('Access-Control-Allow-Methods: GET, POST, PUT, PATCH, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Content-Type: application/json; charset=UTF-8');

// Security headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');
if (APP_ENV === 'production') {
    header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
}

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Parse request
$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$uri = rtrim($uri, '/') ?: '/';

// Get JSON body
$body = json_decode(file_get_contents('php://input'), true) ?? [];

use CashBook\Core\Router;
use CashBook\Core\Request;

$router = new Router();
$request = new Request($method, $uri, $body, getallheaders(), $_GET);

// Load route definitions
require __DIR__ . '/../routes/api.php';

// Dispatch
try {
    $router->dispatch($request);
} catch (\Exception $e) {
    $statusCode = $e->getCode() >= 400 && $e->getCode() < 600 ? $e->getCode() : 500;
    http_response_code($statusCode);

    // Log all 500-level errors
    if ($statusCode >= 500) {
        error_log(sprintf(
            "[CashBook ERR] %s in %s:%d\n%s",
            $e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString()
        ));
    }

    // Never leak internals in production
    $response = ['success' => false, 'message' => 'Internal Server Error'];
    if (APP_DEBUG) {
        $response['message'] = $e->getMessage();
        $response['error'] = [
            'file' => basename($e->getFile()),
            'line' => $e->getLine(),
        ];
    }
    echo json_encode($response);
}
