# CashBook ERP on cPanel - Getting Started

## Overview

Your CashBook ERP application has been **production-optimized for cPanel hosting**. 

**Key Points**:
- ✅ Frontend: Static HTML/CSS/JS (no Node.js needed)
- ✅ Backend: PHP API on Apache (no Docker needed)
- ✅ Database: PostgreSQL (single domain setup)
- ✅ All on same domain: `https://cashbook.yourcompany.com`
- ✅ Ready to deploy in < 30 minutes

---

## What Was Changed For cPanel?

| Component | Change | Reason |
|-----------|--------|--------|
| Frontend Build | `output: 'export'` | Creates static HTML files instead of Node.js server |
| Frontend Server | None needed | cPanel Apache serves static files |
| Backend | Unchanged | Already pure PHP with no Docker dependency |
| Database | Schema optimized | Removes PostgreSQL extensions not available on cPanel |
| UUID Generation | PHP fallback | Works without `uuid-ossp` extension |
| Routing | Apache .htaccess | Used instead of nginx reverse proxy |
| Configuration | Simplified | Single .env file in `public_html/api/` |

---

## Files You Need to Know About

### Documentation
- **`CPANEL_READY.md`** - This deployment summary
- **`CPANEL_DEPLOYMENT.md`** - Detailed step-by-step deployment guide (10+ pages)
- **`CPANEL_CHECKLIST.md`** - Pre/post deployment verification checklist

### Configuration
- **`frontend/next.config.js`** - Static export enabled
- **`backend/.env.cpanel.example`** - Template for secrets
- **`.htaccess.frontend`** - Frontend routing rules (rename to `.htaccess`)
- **`backend/.htaccess`** - Backend routing rules

### Database
- **`backend/database/schema.cpanel.sql`** - cPanel-compatible schema (import this into PostgreSQL)

### Deployment Scripts
- **`deploy-cpanel.sh`** - For Linux/macOS (optional automation)
- **`deploy-cpanel.bat`** - For Windows (optional automation)

### Generated Frontend
- **`frontend/out/`** - Static HTML/CSS/JS files (upload to `public_html/`)

---

## 3-Step Quick Start

### Step 1: Build Frontend
```bash
cd frontend
npm run export
```
This creates `frontend/out/` with all static files.

### Step 2: Upload to cPanel
1. Log into cPanel at `https://your-domain.com:2083`
2. Go to **File Manager**
3. Upload `frontend/out/*` to `public_html/`
4. Upload `backend/*` to `public_html/api/` (skip Dockerfile, docker/)
5. Copy `.env.cpanel.example` to `public_html/api/.env` and fill in secrets

### Step 3: Set Up Database
1. In cPanel **PostgreSQL Databases**: Create `cashbook_prod` database
2. In **PhpPgAdmin**: Import `backend/database/schema.cpanel.sql`
3. Test: `https://your-domain.com/api/health` should return JSON

---

## Architecture Diagram

```
User Browser
    ↓
[https://cashbook.yourcompany.com]
    ↓
Apache (cPanel)
    ├─ Port 80/443
    │
    ├─ /                    → public_html/index.html (Next.js SPA)
    ├─ /login              → public_html/login/index.html
    ├─ /dashboard/…        → public_html/dashboard/index.html
    │
    └─ /api/auth/*         → public_html/api/public/index.php
       /api/invoices/*     → (PHP API handlers)
       /api/…/*            → (via .htaccess rewrite)
            ↓
         Database
         (PostgreSQL cashbook_prod)
```

**Same Domain**: Frontend and backend at same URL → No CORS issues ✅

---

## Command Reference

### Build Frontend (Local Machine)
```bash
cd frontend
npm install
npm run export
# Output: frontend/out/
```

### List All cPanel Config Files
```bash
find . -name "CPANEL*" -o -name ".htaccess*" -o -name "*.cpanel*"
```

### Import Database Schema (via SSH)
```bash
cd ~/public_html/api
psql -U cashbook_user -d cashbook_prod < ../database/schema.cpanel.sql
```

### Test Backend (from anywhere)
```bash
curl https://your-domain.com/api/health
# Returns: {"status":"ok",...}

curl -X POST https://your-domain.com/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{"company_name":"Test","tin":"C0000000000","name":"Admin","email":"admin@test.com","password":"Pass123!"}'
```

### Check Error Logs
```bash
# Frontend errors
tail -f ~/public_html/error_log

# Backend errors
tail -f ~/public_html/api/error_log
```

---

## Critical Configuration Values

Update these in `.env` (located at `public_html/api/.env`):

```env
# Your actual domain
APP_URL=https://cashbook.yourcompany.com
FRONTEND_URL=https://cashbook.yourcompany.com

# Database credentials from cPanel
DB_HOST=localhost
DB_NAME=cashbook_prod
DB_USER=cashbook_user
DB_PASS=YOUR_CPANEL_DB_PASSWORD

# Generate ONCE and NEVER change
JWT_SECRET=$(openssl rand -base64 48)  # 64+ character random string

# These rarely need changes
JWT_EXPIRY=3600              # 1 hour
JWT_REFRESH_EXPIRY=604800    # 7 days
```

---

## Expected Directory Structure After Deployment

```
your-domain.com
│
├── public_html/                   (Frontend - Static HTML)
│   ├── index.html
│   ├── login/index.html
│   ├── dashboard/index.html
│   ├── register/index.html
│   ├── _next/static/               (Bundled CSS/JS)
│   ├── _next/data/                 (Prerendered data)
│   └── .htaccess                   (SPA routing)
│
├── public_html/api/                (Backend - PHP API)
│   ├── public/                     (Web root)
│   │   └── index.php               (API entry point)
│   ├── src/                        (PHP code)
│   │   ├── Controllers/
│   │   ├── Core/
│   │   ├── Middleware/
│   │   └── Util/
│   ├── config/                     (Configuration)
│   │   ├── app.php
│   │   └── database.cpanel.php
│   ├── routes/
│   │   └── api.php
│   ├── vendor/                     (Composer dependencies)
│   ├── .env                        (Secrets - don't commit!)
│   ├── .htaccess                   (API routing)
│   ├── composer.json
│   └── composer.lock
│
└── (PostgreSQL Database)
    ├── Database: cashbook_prod
    ├── User: cashbook_user
    └── Tables: 30+ (accounts, invoices, payroll, etc.)
```

---

## What You Don't Need Anymore

❌ Docker / Docker Compose  
❌ nginx (Apache handles it)  
❌ Node.js in production  
❌ Supervisor / PM2  
❌ Load balancers (unless very high traffic)  

---

## Performance Tips

### 1. Enable Caching
Already configured in `.htaccess`:
- Static assets (.js, .css, .png) → 1-year cache
- HTML → no cache (always fresh)

### 2. Enable Gzip
Already configured in `.htaccess`:
- Reduces bandwidth by 60-80%

### 3. Monitor Database
```bash
# Check database size
psql -U cashbook_user -d cashbook_prod -c "SELECT pg_size_pretty(pg_database_size('cashbook_prod'));"

# Count rows
psql -U cashbook_user -d cashbook_prod -c "SELECT COUNT(*) FROM invoices;"
```

### 4. Archive Old Data
```bash
# Archive audit logs older than 6 months
psql -U cashbook_user -d cashbook_prod -c "DELETE FROM audit_log WHERE created_at < NOW() - INTERVAL '6 months';"
```

---

## Troubleshooting Quick Links

| Problem | Solution |
|---------|----------|
| Frontend shows blank page | Check `.htaccess` exists, enable mod_rewrite |
| API returns 500 error | Check error logs (`tail -f error_log`), verify `.env` |
| Can't connect to database | Verify credentials in `.env`, check PostgreSQL is running |
| CORS errors in browser console | Should NOT happen - both on same domain. Check origin header. |
| UUID function not found | Code has fallback to PHP UUIDs. Should work automatically. |
| Uploads failing | Set `chmod 777 public_html/api/public/uploads/` |
| Slow site performance | Check database size, review error logs, enable caching |

**More help**: See `CPANEL_DEPLOYMENT.md` Troubleshooting section.

---

## Security Reminders

1. **Change `JWT_SECRET`** - Generate a new one with `openssl rand -base64 48`
2. **Change DB password** - Use a strong 20+ character password
3. **Enable HTTPS** - Set up Let's Encrypt in cPanel
4. **Disable directory listing** - You should run `Options -Indexes` in .htaccess (already done)
5. **Regular backups** - Database and files
6. **Monitor logs** - Check `error_log` weekly

---

## Support

### Documentation
- **Complete Guide**: `CPANEL_DEPLOYMENT.md` (detailed instructions)
- **Checklist**: `CPANEL_CHECKLIST.md` (pre/post deployment validation)
- **This Guide**: `CPANEL_READY.md` (quick reference)

### If You Get Stuck

1. Check the **Error Logs**:
   ```bash
   tail -f ~/public_html/error_log
   tail -f ~/public_html/api/error_log
   ```

2. Check **Database Connection**:
   ```bash
   psql -U cashbook_user -d cashbook_prod -c "SELECT 1;"
   ```

3. Test **API Health**:
   ```bash
   curl -v https://your-domain.com/api/health
   ```

4. Read **Troubleshooting Section** in `CPANEL_DEPLOYMENT.md`

---

## Next Steps

1. **Read**: `CPANEL_DEPLOYMENT.md` for detailed instructions
2. **Checklist**: Use `CPANEL_CHECKLIST.md` to track progress
3. **Execute**: Follow the 8-step deployment guide
4. **Test**: Verify frontend loads and API responds
5. **Monitor**: Check error logs for the first week

---

**Ready to deploy? Start with `CPANEL_DEPLOYMENT.md`!**

📄 **CPANEL_DEPLOYMENT.md** - Complete step-by-step guide  
✅ **CPANEL_CHECKLIST.md** - Track your progress  
🚀 **You're 30 minutes away from production!**
